# -----------------------------------------------------------------------------
# Programm         : G r a p h . p m
# -----------------------------------------------------------------------------
# Autor            : Christian Dhl
# Erstellt         : 08. Februar 2004
# Zuletzt gendert : 10. Februar 2004
# Aufgabe          : Logische Graphenklasse (ohne Darstellungselemente).
# -----------------------------------------------------------------------------

# -----------------------------------------------------------------------------
# Globale private Datei-Variablen:
# -----------------------------------------------------------------------------
my $Debug  = 2; # Debug level


# -----------------------------------------------------------------------------
# Klasse           : G r a p h
# abgeleitet von   : none
# -----------------------------------------------------------------------------
# Autor            : Christian Dhl
# Beschreibung     : Eine Graphen-Klasse.
# Attribute        : n           - Anzahl der Ecken
#                    m           - Anzahl der Kanten
#                    kantenform  - Art der Kante (normal, Pfeil hin...)
#                    _kanten     - anonymes Array von Kante-Objekten
# Klassen-Methoden : new         - Erstellt ein Objekt dieser Klasse.
# Objekt-Methoden  : _property         - Hilfsfunktion fr Getter / Setter
#                    n                 - Getter / Setter
#                    m                 - Getter / Setter
#                    kantenform        - Getter / Setter
#                    _kanten           - Interner Getter / Setter
#                    setze_kanten      - Ersetzt die Kantenmenge
#                    loesche_kanten    - Lscht die Kantenmenge
#                    kante             - Getter/Setter ber Index fr
#                                        eine Kante aus der Menge
#                    entnehme_kante    - spliced eine Kante heraus
#                    fuege_kante_hinzu - pushed eine Kante hinzu
#                    ecke_einfuegen    - Ecke einfgen
#                    ecke_loeschen     - Ecke lschen
#                    ecke_mergen       - Ecke mergen (Ecke wird gelscht
#                                        und die Nachbarn der Ecke alle
#                                        miteinander durch Kanten verbunden).
#                    kante_existiert   - Testet, ob die beiden angegeben Ecken
#                                        durch eine Kante verbunden sind.
#                    kante_einfuegen   - Fgt eine Kante ein.
#                    kante_loeschen    - Lscht eine Kante.
#                    clear             - Lscht alles.
# -----------------------------------------------------------------------------
# 0.0.1 - 08.02.2004 - CD - erstellt
# -----------------------------------------------------------------------------
package Graph::GED::Graph;

use strict;
use warnings;

use Data::Dumper;

use Graph::GED::Kante;

sub new {
# -----------------------------------------------------------------------------
# Methode      : n e w
# Klasse       : Graph
# -----------------------------------------------------------------------------
# Autor        : Christian Dhl
# Aufgabe      : Erzeugt ein neues Graph-Objekt (oder ein Objekt einer Klasse,
#                die von dieser Klasse abgeleitet wurde und uns bittet, das
#                Objekt fr sie zu erzeugen).
# Parameter    : 1) Klasse
#                2) ev. gewnschte Klasse (im Fall einer abgeleiteten Klasse)
#                Und ggf. Attribute: siehe Klassenbeschreibung oben
# Rckgabewert : Das geblesste neue Objekt.
# -----------------------------------------------------------------------------
# 0.0.1 - 08.02.2004 - CD - erstellt
# -----------------------------------------------------------------------------

    my $class = shift; # Die Klasse, von der new aufgerufen wurde:
                       #     my $object = Graph->new(...);
                       # oder
                       #     my $object = new Graph(...);


    my $wantclass;     # Dies ist die Klasse, in welche das neue Objekt ge-
                       # blessed werden soll.
                       # Falls die Methode new mit einer gerade Anzahl an
                       # Parametern aufgerufen wurde (nachdem $class schon ent-
                       # nommen wurde), wurde keine gewnschte Klasse angege-
                       # ben. In diesem Fall verwenden wir $class.
                       # Anderenfalls verwenden wir die bergebene Klasse.

    my $n = scalar @_; # Bestimmt die Anzahl an verbliebenen Parametern.

    #--------------------------------------------------------------------------
    # Falls noch Parameter da sind und ihre Anzahl ungerade ist, dann mchte
    # der Aufrufer, dass wir das Objekt auf die gewnschte Klasse blessen, die
    # als nchster Parameter angegeben ist.
    # Also holen wir sie mit shift aus den Aufrufparametern und legen sie
    # in $wantclass ab.
    #--------------------------------------------------------------------------
    if ($n > 0 and ($n >> 1) << 1 != $n) {
        $wantclass = shift;
    }
    #--------------------------------------------------------------------------
    # Anderenfalls wurde keine gewnschte Klasse angegeben und wir verwenden
    # $class aus dem ersten Parameter:
    #--------------------------------------------------------------------------
    else {
        $wantclass = $class;
    }

    #--------------------------------------------------------------------------
    # Ausgabe der Aufrufparameter:
    #--------------------------------------------------------------------------
    print "Graph::GED::Graph::new\n",
          "\tclass      = '$class'\n",
          "\twant class = '$wantclass'\n",
          "\tn          = $n\n",
          "\tParameter  = @_\n",
          if $Debug > 1;

    #--------------------------------------------------------------------------
    # Wir erzeugen ein Objekt in der gewnschten Klasse (eine geblessde Refe-
    # renz auf einen ev. leeren anonymen Hash):
    #--------------------------------------------------------------------------
    my $self = bless({@_}, $wantclass);

    #--------------------------------------------------------------------------
    # Nun setzen wir noch ein paar Attribute:
    #--------------------------------------------------------------------------
    $self->n         (0);
    $self->m         (0);
    $self->kantenform('');
    $self->_kanten   ([]);

    #--------------------------------------------------------------------------
    # ... und geben das erzeugte Objekt zurck:
    #--------------------------------------------------------------------------
    return $self;

} # sub Graph::new


# -----------------------------------------------------------------------------
# Methode      : _ p r o p e r t y
# Klasse       : Graph
# -----------------------------------------------------------------------------
# Autor        : Christian Dhl
# Aufgabe      : Private Funktion fr die einfache Definition von Gettern und
#                Settern.
# Parameter    : Attribut, Parameter
# Rckgabewert : Der aktuelle Wert (im Falle des Getters) oder der alte Wert
#                (im falle des Setters).
# -----------------------------------------------------------------------------
# 0.0.1 - 08.02.2004 - CD - erstellt
# -----------------------------------------------------------------------------
sub _property ($$;$) {
    my $self   = shift;    # Objekt
    my $attr   = shift;    # Attribute, das gesetzt oder geholt wird
    my $setter = @_ == 1;  # Ist die Methode Setter?
    my $value  = shift;    # Zu setzender Wert, falls Setter

    # Falls man hier "if (defined $value)" benutzen wrde, so knnte man keine
    # undefinierten Werte setzen, deshalb verwenden wir diese Form:
    if ($setter) {
        my $old_value = $self->{$attr};
        $self->{$attr} = $value;
        return $old_value;
    }
    else {
        return $self->{$attr};
    }
} # sub Graph::_property


# -----------------------------------------------------------------------------
# The getter/setter methods:
# -----------------------------------------------------------------------------
sub n          { return shift->_property('n',          @_) }
sub m          { return shift->_property('m',          @_) }
sub kantenform { return shift->_property('kantenform', @_) }
sub _kanten    { return shift->_property('_kanten',    @_) }

# -----------------------------------------------------------------------------
# Hilfsfunktionen fr n und m:
# -----------------------------------------------------------------------------
sub inc_n { $_[0]->n($_[0]->n()+1) } # n bzw. m
sub dec_n { $_[0]->n($_[0]->n()-1) } # in-
sub inc_m { $_[0]->m($_[0]->m()+1) } # bzw.
sub dec_m { $_[0]->m($_[0]->m()-1) } # decrementieren



# =============================================================================
#
#
#   H I L F S M E T H O D E N
#
#   (nur diese verwenden _kanten)
#
# =============================================================================


sub setze_kanten {
# -----------------------------------------------------------------------------
# Methode      : s e t z e _ k a n t e n
# Klasse       : Graph
# -----------------------------------------------------------------------------
# Autor        : Christian Dhl
# Aufgabe      : Ersetzt die gesamte Kantenmenge.
#                Die alte Menge wird vorher durch das Setzen auf undef
#                freigegeben
# Parameter    : Referenz auf Array (mit Kante-Objekten)
# Rckgabewert : keiner
# -----------------------------------------------------------------------------
# 0.0.1 - 10.02.2004 - CD - erstellt
# -----------------------------------------------------------------------------

    $_[0]->_kanten(undef);
    $_[0]->_kanten($_[1]);

} # sub setze_kanten


sub loesche_kanten {
# -----------------------------------------------------------------------------
# Methode      : l o e s c h e _ k a n t e n
# Klasse       : Graph
# -----------------------------------------------------------------------------
# Autor        : Christian Dhl
# Aufgabe      : Setzt die gesamte Kantenmenge auf eine leere Menge.
#                Die alte Menge wird vorher durch das Setzen auf undef
#                freigegeben
# Parameter    : keine
# Rckgabewert : keiner
# -----------------------------------------------------------------------------
# 0.0.1 - 10.02.2004 - CD - erstellt
# -----------------------------------------------------------------------------

    $_[0]->_kanten(undef);
    $_[0]->_kanten([]);

} # sub loesche_kanten


sub kante {
# -----------------------------------------------------------------------------
# Methode      : k a n t e
# Klasse       : Graph
# -----------------------------------------------------------------------------
# Autor        : Christian Dhl
# Aufgabe      : Setter / Getter fr ein Kante(nobjekt)
# Parameter    : 1) Index
#                2) Wert (optional)
# Rckgabewert : Der aktuelle Wert (im Falle des Getters) oder der alte Wert
#                (im falle des Setters).
# -----------------------------------------------------------------------------
# 0.0.1 - 10.02.2004 - CD - erstellt
# -----------------------------------------------------------------------------

    if (@_ == 3) {
        my $aw = $_[0]->_kanten()->[$_[1]];
        $_[0]->_kanten()->[$_[1]] = $_[2];
        return $aw;
    }
    else {
        return $_[0]->_kanten()->[$_[1]];
    }

} # sub kante


sub entnehme_kante {
# -----------------------------------------------------------------------------
# Methode      : e n t n e h m e _ k a n t e
# Klasse       : Graph
# -----------------------------------------------------------------------------
# Autor        : Christian Dhl
# Aufgabe      : Entfernt eine Kante aus der Kantenmenge und gibt diese zurck.
# Parameter    : Index
# Rckgabewert : keiner
# -----------------------------------------------------------------------------
# 0.0.1 - 10.02.2004 - CD - erstellt
# -----------------------------------------------------------------------------

    return splice @{$_[0]->_kanten()}, $_[1], 1;

} # sub entnehme_kante


sub fuege_kante_hinzu {
# -----------------------------------------------------------------------------
# Methode      : f u e g e _ k a n t e _ h i n z u
# Klasse       : Graph
# -----------------------------------------------------------------------------
# Autor        : Christian Dhl
# Aufgabe      : Fgt eine Kante zur Kantenmenge hinzu
# Parameter    : Kante-Objekt
# Rckgabewert : keiner
# -----------------------------------------------------------------------------
# 0.0.1 - 10.02.2004 - CD - erstellt
# -----------------------------------------------------------------------------

    push @{$_[0]->_kanten()}, $_[1];

} # sub fuege_kante_hinzu



# =============================================================================
#
#
#   M E T H O D E N
#
#
# =============================================================================


sub ecke_einfuegen {
# -----------------------------------------------------------------------------
# Methode      : e c k e _ e i n f u e g e n
# Klasse       : Graph
# -----------------------------------------------------------------------------
# Autor        : Christian Dhl
# Aufgabe      : Fgt eine neue Ecke in den Graphen ein.
# Parameter    : keine
# Rckgabewert : keiner
# -----------------------------------------------------------------------------
# 0.0.1 - 08.02.2004 - CD - erstellt
# 0.0.2 - 10.02.2004 - CD - inc_n verwendet
# -----------------------------------------------------------------------------

    my $self = shift;

    die "Graph::GED::Graph::ecke_einfuegen mit Parametern aufgerufen!" if scalar @_;

    print "Graph::GED::Graph::ecke_einfuegen\n" if $Debug;

    #--------------------------------------------------------------------------
    # Eckenzhler incrementieren
    #--------------------------------------------------------------------------
    $self->inc_n();

} # sub Graph::ecke_einfuegen


sub ecke_loeschen {
# -----------------------------------------------------------------------------
# Methode      : e c k e _ l o e s c h e n
# Klasse       : Graph
# -----------------------------------------------------------------------------
# Autor        : Christian Dhl
# Aufgabe      : Lscht die angegebene Ecke mit allen inzidierenden Kanten
#                aus dem Graphen.
# Parameter    : zu lschende Ecke
# Rckgabewert : keiner
# -----------------------------------------------------------------------------
# 0.0.1 - 08.02.2004 - CD - erstellt
# 0.0.2 - 10.02.2004 - CD - neue Hilfsmethoden verwendet.
#                           Umstellung auf die Kantenklasse.
# -----------------------------------------------------------------------------

    my $self = shift;

    die "Graph::GED::Graph::ecke_loeschen falscher Anzahl Parametern aufgerufen!"
        if scalar @_ != 1;
    die "Graph::GED::Graph::ecke_loeschen aufgerufen, aber der Graph hat keine Ecken!"
        unless $self->n() > 0;

    my $e = shift;


    print "Graph::GED::Graph::ecke_loeschen($e)\n" if $Debug;

    die "Graph::GED::Graph::ecke_loeschen mit falschem Eckenwert '$e' aufgerufen, n ist " .
        $self->n() ."!"
        if $e > $self->n();

    die "Interner Fehler!" if scalar @{$self->_kanten()} != $self->m();

    print Dumper($self) if $Debug > 1;

    #--------------------------------------------------------------------------
    # Kanten raussuchen, die zu dieser Ecke gehren und lschen:
    #--------------------------------------------------------------------------
    for my $i (reverse 0..$self->m()-1) {
        if ($self->kante($i)->inzidiert_mit_ecke($e)) {
            $self->kante_loeschen($i);
        }
    }

    #--------------------------------------------------------------------------
    # Eckenzhler decrementieren
    #--------------------------------------------------------------------------
    $self->dec_n();

    #--------------------------------------------------------------------------
    # Korrigieren der hheren Eckennummern:
    #--------------------------------------------------------------------------
    for my $i (0..$self->m()-1) {
        my $k = $self->kante($i);
        $k->a($k->a()-1) if $k->a() >= $e;
        $k->e($k->e()-1) if $k->e() >= $e;
    }

    print Dumper($self) if $Debug > 1;
} # sub ecke_loeschen


sub ecke_mergen {
# -----------------------------------------------------------------------------
# Methode      : e c k e _ m e r g e n
# Klasse       : Graph
# -----------------------------------------------------------------------------
# Autor        : Christian Dhl
# Aufgabe      : Mergt die angegebene Ecke (d.h. die Ecke wird gelscht und die
#                Kanten zusammengezogen.
# Parameter    : Zu mergende Ecke
# Rckgabewert : keiner
# -----------------------------------------------------------------------------
# 0.0.1 - 08.02.2004 - CD - erstellt
# 0.0.2 - 10.02.2004 - CD - neue Hilfsmethoden verwendet.
#                           Umstellung auf die Kantenklasse.
# -----------------------------------------------------------------------------

    my $self = shift;

    die "Graph::GED::Graph::ecke_mergen mit falscher Anzahl Parametern aufgerufen!"
        if scalar @_ != 1;

    my $e = shift;

    print "Graph::GED::Graph::ecke_mergen($e)\n" if $Debug;

    die "Interner Fehler!" if scalar @{$self->_kanten()} != $self->m();

    #--------------------------------------------------------------------------
    # Ecke mergen:
    #--------------------------------------------------------------------------
    print Dumper($self) if $Debug > 1;

    #--------------------------------------------------------------------------
    # Kanten raussuchen, die zu dieser Ecke gehren:
    #--------------------------------------------------------------------------
    my @ecken = (); # Nachbarn von $e, die dann verbunden werden mssen
    my @kform = (); # Kantenformen, im Moment ignoriert

    for my $i (reverse 0..$self->m()-1) {
        if ($self->kante($i)->inzidiert_mit_ecke($e)) {
            my $k = $self->entnehme_kante($i);

            print "loesche Kante $i ", $k->a(), " --- ", $k->e(), "\n"
                  if $Debug>1;

            push @ecken, ($k->a()!=$e ? $k->a() : $k->e());
            push @kform, $k->form();
            # Bei der Kantenform msste man noch bercksichtigen, welche Ecke
            # die Kante $e war. Und dann bei Kantenpaaren mit der selben
            # Ausprgung diese wieder verwenden. Im Moment wird alles als
            # normale Kante erzeugt.

            $self->dec_m();
        }
    }

    print "Nachbarn der geloeschten Ecke: @ecken\n";

    #--------------------------------------------------------------------------
    # Neue Kanten eintragen:
    #--------------------------------------------------------------------------
    for my $i1 (0..$#ecken-1) {
        print "i1=$i1\n";
        for my $i2 ($i1+1..$#ecken) {
            print "i2=$i2\n";
            print "fuege Kante ein: $i1 --- $i2\n" if $Debug;
            $self->kante_einfuegen($ecken[$i1], $ecken[$i2])
                unless $self->kante_existiert($ecken[$i1], $ecken[$i2]);
        }
    }

    #--------------------------------------------------------------------------
    # Ecke lschen:
    #--------------------------------------------------------------------------
    $self->dec_n();

    #--------------------------------------------------------------------------
    # Korrigieren der hheren Eckennummern:
    #--------------------------------------------------------------------------
    for my $i (0..$self->m()-1) {
        my $k = $self->kante($i);
        $k->a($k->a()-1) if $k->a() >= $e;
        $k->e($k->e()-1) if $k->e() >= $e;
    }

    print Dumper($self) if $Debug > 1;

} # sub Graph::ecke_mergen


sub kante_existiert {
# -----------------------------------------------------------------------------
# Methode      : k a n t e _ e x i s t i e r t
# Klasse       : Graph
# -----------------------------------------------------------------------------
# Autor        : Christian Dhl
# Aufgabe      : Testet, ob die angegebene Kante existiert
# Parameter    : 1) Ecke Anfang
#                2) Ecke Ende
# Rckgabewert : undef falls nicht vorhanden, anderenfalls Kantentyp
# -----------------------------------------------------------------------------
# 0.0.1 - 08.02.2004 - CD - erstellt
# 0.0.2 - 10.02.2004 - CD - neue Hilfsmethoden verwendet.
#                           Umstellung auf die Kantenklasse.
# -----------------------------------------------------------------------------

    my $self = shift;

    die "Graph::GED::Graph::kante_existiert mit falscher Anzahl Parametern aufgerufen!"
        if scalar @_ != 2;

    my $ea   = shift;
    my $ee   = shift;

    my $form = undef;

    print "Graph::GED::Graph::kante_existiert($ea, $ee)\n" if $Debug;

    die "Interner Fehler!" if scalar @{$self->_kanten()} != $self->m();

    for my $i (0..$self->m()-1) {
        my $k = $self->kante($i);
        if (
            $ea == $k->a() and $ee == $k->e() or
            $ea == $k->e() and $ee == $k->a()
           )
        {
            $form = $k->form();
            last;
        }
    }


    return $form;

} # sub Graph::kante_existiert


sub kante_einfuegen {
# -----------------------------------------------------------------------------
# Methode      : k a n t e _ e i n f u e g e n
# Klasse       : Graph
# -----------------------------------------------------------------------------
# Autor        : Christian Dhl
# Aufgabe      : Fgt eine Kante in den Graphen ein.
# Parameter    : 1) Startecke
#                2) Zielecke
#                3) Optional Kantenform, sonst $self->kantenform()
# Rckgabewert : keiner
# -----------------------------------------------------------------------------
# 0.0.1 - 08.02.2004 - CD - erstellt
# 0.0.2 - 10.02.2004 - CD - neue Hilfsmethoden verwendet.
#                           Umstellung auf die Kantenklasse.
# -----------------------------------------------------------------------------

    my $self = shift;

    die "Graph::GED::Graph::kante_einfuegen falscher Anzahl Parametern aufgerufen!"
        if scalar @_ < 2 or scalar @_ > 3;
    die "Graph::GED::Graph::kante_einfuegen aufgerufen, aber der Graph hat keine Ecken!"
        unless $self->n() > 0;

    my $ea = shift;
    my $ee = shift;
    my $kf = shift || $self->kantenform();


    print "Graph::GED::Graph::kante_einfuegen($ea, $ee, $kf)\n" if $Debug;

    my $k = new Graph::GED::Kante;
    $k->a   ($ea);
    $k->e   ($ee);
    $k->form($kf);

    $self->fuege_kante_hinzu($k);

    $self->inc_m();

} # sub Graph::kante_einfuegen


sub kante_loeschen {
# -----------------------------------------------------------------------------
# Methode      : k a n t e _ l o e s c h e n
# Klasse       : Graph
# -----------------------------------------------------------------------------
# Autor        : Christian Dhl
# Aufgabe      : Lscht die angegebene Kante aus dem Graphen.
# Parameter    : Kante
# Rckgabewert : keiner
# -----------------------------------------------------------------------------
# 0.0.1 - 08.02.2004 - CD - erstellt
# 0.0.2 - 10.02.2004 - CD - neue Hilfsmethoden verwendet.
#                           Umstellung auf die Kantenklasse.
# -----------------------------------------------------------------------------

    my $self = shift;

    die "Graph::GED::Graph::kante_loeschen mit falscher Anzahl Parametern aufgerufen!"
        if scalar @_ != 1;

    my $ki   = shift;

    print "Graph::GED::Graph::kante_loeschen($ki)\n" if $Debug;

    my $k = $self->entnehme_kante($ki);

    print "lsche Kante $ki ", $k->a(), " --- ", $k->e(), "\n"
          if $Debug>1;


    $self->dec_m();

} # sub Graph::kante_loeschen


sub clear {
# -----------------------------------------------------------------------------
# Methode      : c l e a r
# Klasse       : Graph
# -----------------------------------------------------------------------------
# Autor        : Christian Dhl
# Aufgabe      : Lscht alle Ecken und Kanten.
# Parameter    : keine
# Rckgabewert : keiner
# -----------------------------------------------------------------------------
# 0.0.1 - 09.02.2003 - CD - Erstellt
# 0.0.2 - 10.02.2004 - CD - neue Hilfsmethoden verwendet.
#                           Umstellung auf die Kantenklasse.
# -----------------------------------------------------------------------------

    my $self = shift;

    die "Graph::GED::Graph::clear mit Parametern aufgerufen!"
        if scalar @_ > 0;

    print "Graph::GED::Graph::clear\n" if $Debug;

    #--------------------------------------------------------------------------
    # Ecken- und Kantenzahl auf Null setzen:
    #--------------------------------------------------------------------------
    $self->n(0);
    $self->m(0);

    #--------------------------------------------------------------------------
    # Kanten lschen:
    #--------------------------------------------------------------------------
    $self->loesche_kanten();
} # sub Graph2d::clear


return 1;

__END__

=head1 NAME

Graph.pm - a module for graphs


=head1 COPYRIGHT

Copyright 2004 Christian Duehl. All rights reserved.

This library is free software. You can redistribute it and/or
modify it under the same terms as perl itself.

=cut
