package de.duehl.basics.io.textfile;

/*
 * Copyright 2021 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.util.List;

import de.duehl.basics.io.textfile.data.Definition;
import de.duehl.basics.io.textfile.data.Definitions;
import de.duehl.basics.text.Text;

/**
 * Diese Klasse liest eine Liste mit den Texten im Format von Zuweisungen mit Gleichheitszeichen
 * oder Doppelpunkt aus den Zeilen einer Textdatei ein.
 *
 * Das Format sieht wie folgt aus:
 *
 *     Schlüssel : Wert
 *
 * oder
 *
 *     Schlüssel = Wert
 *
 * Kommt in einer Datenzeile weiter hinten das Kommentarzeichen vor, so wird ab dort abgeschnitten
 * und getrimmt.
 *
 * @version 1.01     2021-03-01
 * @author Christian Dühl
 */

public class DefinitionsFromTextFileReader {

    private final static String SEPARATING_REGEX = " *[=:] *";

    /** Der zum eigentlichen Einlesen verwendete Reader. */
    private final StringsFromTextFileReader reader;

    /**
     * Konstruktor.
     *
     * @param reader
     *            Der zum eigentlichen Einlesen verwendete Reader.
     */
    public DefinitionsFromTextFileReader(StringsFromTextFileReader reader) {
        this.reader = reader;
        reader.doNotTrimLines();
    }

    /** Legt fest, dass der Reader ruhig sein soll. */
    public void beQuiet() {
        reader.beQuiet();
    }

    /**
     * Gibt an, dass die eingelesenen Zeilen in Kleinbuchstaben gewandelt werden.
     *
     * Hierbei wird Locale.GERMAN eingesetzt.
     */
    public void changeToLowerCase() {
        reader.changeToLowerCase();
    }

    /**
     * Liest die Listen aus der Datei ein. Das erste Wort in der Liste ist die Vokabel und die
     * anderen die Übersetzungen.
     */
    public Definitions read() {
        Definitions definitions = new Definitions();

        for (String line : reader.read()) {
            List<String> parts = Text.splitBy(line, SEPARATING_REGEX);
            if (parts.size() != 2) {
                throw new IllegalArgumentException("Zeile '" + line
                        + "' lässt sich nicht an ':' oder '=' in zwei Teile teilen..");
            }
            else {
                String key = parts.get(0);
                String value = parts.get(1);
                Definition definition = new Definition(key, value);
                definitions.add(definition);
            }
        }

        return definitions;
    }

}
