package de.duehl.basics.text.html;

/*
 * Copyright 2022 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.util.List;

import de.duehl.basics.text.html.generation.SwingHtmlBuilder;

/**
 * Diese Klasse erzeugt eine Beschreibung der übergebenen Daten in Form von HTML.
 *
 * @version 1.01     2022-04-08
 * @author Christian Dühl
 */

public class HtmlDatasetDescriptionGenerator {

    /** Überschriften für die darzustellenden Felder. */
    private final List<String> titles;

    /** Werte der darzustellenden Felder. */
    private final List<String> values;

    /** Größe der beiden Listen. */
    private final int size;

    /** Der Html-Builder, in dem die Darstellung hier erzeugt wird. */
    private SwingHtmlBuilder html;

    private HtmlDatasetDescriptionGenerator(List<String> titles, List<String> values) {
        this.titles = titles;
        this.values = values;
        if (titles.size() != values.size()) {
            throw new IllegalArgumentException("Die beiden übergebenen Listen mit den "
                    + "Überschriften und den Werten haben unterschiedliche Größen!\n"
                    + "\t" + "titles.size() = " + titles.size() + "\n"
                    + "\t" + "values.size() = " + values.size() + "\n");
        }
        size = titles.size();
    }

    public String generateHtmlDatasetDescription() {
        html = new SwingHtmlBuilder();
        html.appendHtml5HeadWithOwnCss("Beschreibung des aktuellen Datensatzes");
        html.appendOpeningTableWithBorderWidth(2);

        appendTableHeader();
        appendTableBody();

        html.appendFoot();
        return html.toString();
    }

    private void appendTableHeader() {
        html.appendOpeningThead();
        html.appendOpeningTr();
        html.appendLeftAlignedTh("Feldname");
        html.appendLeftAlignedTh("Beschreibung");
        html.appendClosingTr();
        html.appendClosingThead();
    }

    private void appendTableBody() {
        html.appendOpeningTbody();

        for (int index = 0; index < size; ++index) {
            String title = titles.get(index);
            String value = values.get(index);
            value = HtmlTool.encodeHtmlAmpLtAndGt(value);
            appendTableBodyRow(title, value);
        }

        html.appendClosingTbody();
    }

    private void appendTableBodyRow(String title, String value) {
        html.appendOpeningTr();
        html.appendTd(title);
        html.appendTd(value);
        html.appendClosingTr();
    }

    /** Erzeugt eine Beschreibung der übergebenen Daten in Form von HTML. */
    public static String generateHtmlDatasetDescription(List<String> titles, List<String> values) {
        HtmlDatasetDescriptionGenerator generator =
                new HtmlDatasetDescriptionGenerator(titles, values);
        return generator.generateHtmlDatasetDescription();
    }

}
