package de.duehl.swing.ui.components.elements;

/*
 * Copyright 2019 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.event.ActionListener;
import java.awt.event.FocusListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;

import javax.swing.JButton;
import javax.swing.JPanel;
import javax.swing.JTextField;

import de.duehl.swing.ui.GuiTools;
import de.duehl.swing.ui.colors.Colorizer;

/**
 * Diese Klasse stellt eine Auswahl von einem frei bearbeitbaren Text mit einem Schalter um diesen
 * zu ändern, etwa in einem Optionendialog dar.
 *
 * @version 1.01     2019-08-09
 * @author Christian Dühl
 */

public class TextFieldWithChangeButton {

    /** Textfeld mit dem Pfad zu der Datei oder dem Verzeichnis. */
    private final JTextField textField;

    /** Button zum ändern des Pfades zur Datei oder zum Verzeichnis. */
    private final JButton changeButton;

    /** Panel mit Überschrift, Pfad und Änderungsbutton. */
    private final Component panel;

    /** Konstruktor. */
    public TextFieldWithChangeButton() {
        textField = createTextField();
        changeButton = createChangeButton();
        panel = createPanel();
    }

    /** Erzeugt das Textfeld. */
    private static JTextField createTextField() {
        JTextField textField = new JTextField();
        GuiTools.setEditFieldColors(textField);
        return textField;
    }

    /** Erzeugt den Button zum Ändern des Pfades. */
    private static JButton createChangeButton() {
        JButton button = new JButton("ändern");
        return button;
    }

    /** Erzeugt den Panel Pfad und Änderungsbutton. */
    private Component createPanel() {
        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());

        panel.add(textField, BorderLayout.CENTER);
        panel.add(changeButton, BorderLayout.EAST);

        return panel;
    }

    /** Fügt einen ActionListener zum "ändern"-Schalter hinzu. */
    public void addButtonActionListener(ActionListener actionListener) {
        changeButton.addActionListener(actionListener);
    }

    /** Fügt einen ActionListener zum Textfeld hinzu. */
    public void addTextFieldActionListener(ActionListener actionListener) {
        textField.addActionListener(actionListener);
    }

    /** Getter für den (getrimmten) Text des Textfeldes. */
    public String getText() {
        String text = textField.getText();
        text = text.trim();
        return text;
    }

    /** Setzt den Text des Textfeldes. */
    public void setText(String text) {
        textField.setText(text);
    }

    /** Aktiviert oder deaktiviert Textfeld und Button. */
    public void setEnabled(boolean selected) {
        textField.setEnabled(selected);
        changeButton.setEnabled(selected);
    }

    /** Färbt alle Komponenten mit dem übergebenen Colorizer ein. */
    public void colorize(Colorizer colorizer) {
        colorizer.setColors(panel);
        colorizer.setColors(textField);
        colorizer.setColors(changeButton);
        colorizer.setEditFieldColors(textField);
    }

    /** Getter für den Panel Pfad und Änderungsbutton. */
    public Component getPanel() {
        return panel;
    }

    /** Setter für die gewünschte Größe des Textfeldes. */
    public void setPreferredSize(Dimension preferredSize) {
        textField.setPreferredSize(preferredSize);
    }

    /** Setter für den auf dem Änderungsbutton angezeigten Text. */
    public void setButtonText(String title) {
        changeButton.setText(title);
    }

    /** Löst ein Anklicken des Buttons aus. */
    public void clickButton() {
        changeButton.doClick();
    }

    /** Setzt die bevorzugt Größe des Buttons. */
    public void setButtonPreferredSize(Dimension buttonDimension) {
        changeButton.setPreferredSize(buttonDimension);
    }

    /** Fügt einen KeyListener zum Textfeld hinzu. */
    public void addKeyListener(KeyListener keyListener) {
        textField.addKeyListener(keyListener);
    }

    /** Fügt einen KeyListener zum Textfeld hinzu, der bei Return den Button auslöst. */
    public void addReturnKeyClickButtonListener() {
        addKeyListener(createClickButtonKeyListener());
    }

    private KeyListener createClickButtonKeyListener() {
        return new KeyListener() {
            @Override
            public void keyTyped(KeyEvent event) {
            }
            @Override
            public void keyReleased(KeyEvent event) {
                int keyCode = event.getKeyCode();
                if (keyCode == KeyEvent.VK_ENTER) {
                    clickButton();
                }
            }
            @Override
            public void keyPressed(KeyEvent event) {
            }
        };
    }

    /** Fügt einen FocusListener zum Textfeld hinzu. */
    public void addFocusListener(FocusListener focusListener) {
        textField.addFocusListener(focusListener);
    }

    /** Setzt den Tooltip. */
    public void setToolTipText(String tooltip) {
        changeButton.setToolTipText(tooltip);
        textField.setToolTipText(tooltip);
    }

}
