package de.duehl.swing.ui.components.selections.datetime;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;

import javax.swing.JPanel;

import de.duehl.basics.datetime.DateAndTime;
import de.duehl.basics.datetime.date.ImmutualDate;
import de.duehl.basics.datetime.time.ImmutualTime;
import de.duehl.swing.ui.components.selections.StringSelection;

/**
 * Diese Klasse stellt eine simple Datum- und Zeiteingabe mit Strings dar.
 *
 * @version 1.01     2020-01-24
 * @author Christian Dühl
 */

public class SimpleDateAndTimeSelection {

    private static final Color IMPORTANT_FIELD_COLOR = new Color(255, 255, 180);

    private final StringSelection dateSelection;
    private final StringSelection timeSelection;
    private final JPanel panel;

    public SimpleDateAndTimeSelection() {
        dateSelection = new StringSelection("Datum");
        timeSelection = new StringSelection("Uhrzeit");

        dateSelection.biggerText(7);
        dateSelection.setBackgroundColor(IMPORTANT_FIELD_COLOR);

        timeSelection.biggerText(7);
        timeSelection.setBackgroundColor(IMPORTANT_FIELD_COLOR);

        panel = createPanel();
    }

    /**
     * Fügt einen KeyListener zu beiden Textfeldern hinzu, der bei Return die übergebene Aktion
     * ausführt.
     */
    public void addReturnListener(Runnable runnable) {
        dateSelection.addReturnListener(runnable);
        timeSelection.addReturnListener(runnable);
    }

    private JPanel createPanel() {
        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());

        panel.add(dateSelection.getPanel(), BorderLayout.NORTH);
        panel.add(timeSelection.getPanel(), BorderLayout.SOUTH);

        return panel;
    }

    public void initDate(String date) {
        dateSelection.setText(date);
    }

    public void initTime(String time) {
        timeSelection.setText(time);
    }

    public void initWithNow() {
        DateAndTime dateAndTime = new DateAndTime();
        initWithDateAndTime(dateAndTime);
    }

    public void initWithDateAndTime(DateAndTime dateAndTime) {
        String date = dateAndTime.getDate().toString();
        String time = dateAndTime.getTime().toString();
        initDate(date);
        initTime(time);
    }

    public void initWithSmoothTimeInAbout10Minutes() {
        DateAndTime dateAndTime = new DateAndTime();
        int minutes = dateAndTime.getTime().getMinute();
        int missing = 10 - (minutes % 10);
        if (missing >= 5) {
            dateAndTime = dateAndTime.addMinutes(missing);
        }
        else {
            dateAndTime = dateAndTime.addMinutes(missing + 5);
        }
        initWithDateAndTime(dateAndTime.setSecondsToZero());
    }

    public Component getPanel() {
        return panel;
    }

    public String getDate() {
        return dateSelection.getTrimmedText().strip();
    }

    public String getTime() {
        return timeSelection.getTrimmedText().strip();
    }

    private String validityCheckFailedCause;

    public boolean checkValidity() {
        validityCheckFailedCause = "";

        String date = getDate();
        String time = getTime();

        ImmutualDate immutualDate = new ImmutualDate(date);
        if (!immutualDate.isValid()) {
            validityCheckFailedCause = "Datumswert nicht valide";
            return false;
        }

        try {
            new ImmutualTime(time);
        }
        catch (IllegalArgumentException exception) {
            validityCheckFailedCause = exception.getMessage();
            return false;
        }

        return true;
    }

    /** Gibt den Grund für das letzte Fehlschlagen von checkValidity() zurück. */
    public String getValidityCheckFailedCause() {
        return validityCheckFailedCause;
    }

    /** Getter für die eingestellten Werte, nur sicher nach checkValidity(). */
    public DateAndTime getDateAndTime() {
        if (checkValidity()) {
            String date = getDate();
            String time = getTime();

            ImmutualDate immutualDate = new ImmutualDate(date);
            ImmutualTime immutualTime = new ImmutualTime(time);
            return new DateAndTime(immutualDate, immutualTime);
        }
        else {
            throw new RuntimeException("Erst die Validität prüfen! - " + validityCheckFailedCause);
        }
    }

}
