package de.duehl.swing.ui.highlightingeditor.buttonbar;

/*
 * Copyright 2017 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.GridLayout;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JPanel;

import de.duehl.swing.ui.resources.IconLoader;
import de.duehl.swing.ui.undoredo.UndoRedoBar;
import de.duehl.basics.history.HistoryStateDisplayer;
import de.duehl.swing.ui.elements.fontsize.FontSizeSelection;
import de.duehl.swing.ui.highlightingeditor.buttonbar.resources.IconDefinitions;

/**
 * Diese Klasse stellt die Zeile mit den Buttons oben über dem Editor dar.
 *
 * @version 1.01     2017-12-19
 * @author Christian Dühl
 */

public class ButtonBar implements HistoryStateDisplayer {

    private final EditorForButtonBar editor;
    private final IconLoader iconLoader;
    private final JPanel panel;
    private final FontSizeSelection fontSizeSelection;
    private final UndoRedoBar undoRedoBar;

    public ButtonBar(EditorForButtonBar editor) {
        this.editor = editor;

        IconDefinitions iconDefinitions = new IconDefinitions();
        iconLoader = iconDefinitions.createIconLoader();
        fontSizeSelection = new FontSizeSelection(editor);
        //fontSizeSelection.createBorder();
        undoRedoBar = new UndoRedoBar(editor);

        panel = new JPanel();

        populatePanel();
    }

    private void populatePanel() {
        panel.setLayout(new BorderLayout());

        panel.add(createFileOperationsPanel(), BorderLayout.WEST);
        fillWithouFilePanel();
    }

    private Component createFileOperationsPanel() {
        JPanel panel = new JPanel();
        panel.setLayout(new GridLayout(1, 0, 2, 2));

        panel.add(createNewFileButton());
        panel.add(createOpenFileButton());
        panel.add(createSaveFileButton());
        panel.add(createSaveAsFileButton());

        return panel;
    }

    public void removeFileOperationsPanel() {
        panel.removeAll();
        fillWithouFilePanel();
    }

    private void fillWithouFilePanel() {
        panel.add(createUndoRedoBar(), BorderLayout.CENTER);
        panel.add(fontSizeSelection.getComponent(), BorderLayout.EAST);
    }

    private Component createUndoRedoBar() {
        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());

        panel.add(undoRedoBar.getComponent(), BorderLayout.WEST);

        return panel;
    }

    private Component createNewFileButton() {
        JButton button = createButton(IconDefinitions.FILE_NEW);
        button.addActionListener(e -> editor.newFile());
        button.setToolTipText("Neue Datei: Leert den Editor und den Dateinamen.");
        return button;
    }

    private Component createOpenFileButton() {
        JButton button = createButton(IconDefinitions.FILE_LOAD);
        button.addActionListener(e -> editor.openFile());
        button.setToolTipText("Datei laden: Liest die Datei neu ein.");
        return button;
    }

    private Component createSaveFileButton() {
        JButton button = createButton(IconDefinitions.FILE_SAVE);
        button.addActionListener(e -> editor.saveFile());
        button.setToolTipText("Datei speichern: Speichet die Datei unter dem hinterlegten "
                + "Dateinamen ab.");
        return button;
    }

    private Component createSaveAsFileButton() {
        JButton button = createButton(IconDefinitions.FILE_SAVE_AS);
        button.addActionListener(e -> editor.saveFileAs());
        button.setToolTipText("Datei speichern unter: Fragt nach einem neuen Dateinamen und "
                + "speichert unter diesem.");
        return button;
    }

    private JButton createButton(String iconName) {
        JButton button = new JButton(iconLoader.loadIcon(iconName));
        button.setBorder(BorderFactory.createRaisedBevelBorder());
        button.setOpaque(true);
        return button;
    }

    /** Erneuert die Anzeige der Schriftgröße aufgrund der im Editor hinterlegten Schriftgröße. */
    public void refreshFontSize() {
        fontSizeSelection.refreshFontSize();
    }

    /** Gibt die Komponente der Button-Bar zurück, um sie in die Gui einzufügen. */
    public Component getComponent() {
        return panel;
    }

    /**
     * Lässt die Benutzung z.B. des Buttons und des Menüeintrags für den vorigen Stand aus der
     * Bearbeitungshistorie zu.
     */
    @Override
    public void enableUndo() {
        undoRedoBar.enableUndo();
    }

    /**
     * Verbietet die Benutzung z.B. des Buttons und des Menüeintrags für den vorigen Stand aus der
     * Bearbeitungshistorie.
     */
    @Override
    public void disableUndo() {
        undoRedoBar.disableUndo();
    }

    /**
     * Lässt die Benutzung z.B. des Buttons und des Menüeintrags für den nächsten Stand aus der
     * Bearbeitungshistorie zu.
     */
    @Override
    public void enableRedo() {
        undoRedoBar.enableRedo();
    }

    /**
     * Verbietet die Benutzung z.B. des Buttons und des Menüeintrags für den nächsten Stand aus der
     * Bearbeitungshistorie.
     */
    @Override
    public void disableRedo() {
        undoRedoBar.disableRedo();
    }

}
