package de.duehl.vocabulary.japanese.data.symbol;

import java.util.ArrayList;
import java.util.List;

import de.duehl.basics.collections.CollectionsHelper;
import de.duehl.basics.text.Text;

/**
 * Diese Klasse stellt die Zeichen der Schrift Hiragana (in UTF-Zeichen) dar.
 *
 * Sie beinhaltet auch Methoden rund darum, um zu fragen, ob eine Zeichenkette ausschließlich
 * aus diesen Zeichen besteht oder wo das erste bzw. letzte dieser Zeichen zu finden ist.
 *
 * @version 1.01     2025-02-02
 * @author Christian Dühl
 */

public enum Hiragana {

    A("あ", "a", KanaSubType.BASIC),
    I("い", "i", KanaSubType.BASIC),
    U("う", "u", KanaSubType.BASIC),
    E("え", "e", KanaSubType.BASIC),
    O("お", "o", KanaSubType.BASIC),

    KA("か", "ka", KanaSubType.BASIC),
    KI("き", "ki", KanaSubType.BASIC),
    KU("く", "ku", KanaSubType.BASIC),
    KE("け", "ke", KanaSubType.BASIC),
    KO("こ", "ko", KanaSubType.BASIC),

    SA("さ", "sa", KanaSubType.BASIC),
    SHI("し", "shi", KanaSubType.BASIC),
    SU("す", "su", KanaSubType.BASIC),
    SE("せ", "se", KanaSubType.BASIC),
    SO("そ", "so", KanaSubType.BASIC),

    TA("た", "ta", KanaSubType.BASIC),
    CHI("ち", "chi", KanaSubType.BASIC),
    TSU("つ", "tsu", KanaSubType.BASIC),
    TE("て", "te", KanaSubType.BASIC),
    TO("と", "to", KanaSubType.BASIC),

    NA("な", "na", KanaSubType.BASIC),
    NI("に", "ni", KanaSubType.BASIC),
    NU("ぬ", "nu", KanaSubType.BASIC),
    NE("ね", "ne", KanaSubType.BASIC),
    NO("の", "no", KanaSubType.BASIC),

    HA("は", "ha", KanaSubType.BASIC),
    HI("ひ", "hi", KanaSubType.BASIC),
    FU("ふ", "fu", KanaSubType.BASIC),
    HE("へ", "he", KanaSubType.BASIC),
    HO("ほ", "ho", KanaSubType.BASIC),

    MA("ま", "ma", KanaSubType.BASIC),
    MI("み", "mi", KanaSubType.BASIC),
    MU("む", "mu", KanaSubType.BASIC),
    ME("め", "me", KanaSubType.BASIC),
    MO("も", "mo", KanaSubType.BASIC),

    YA("や", "ya", KanaSubType.BASIC),
    YU("ゆ", "yu", KanaSubType.BASIC),
    YO("よ", "yo", KanaSubType.BASIC),

    SMALL_YA("ゃ", "kleines ya", KanaSubType.SMALL),
    SMALL_YU("ゅ", "kleines yu", KanaSubType.SMALL),
    SMALL_YO("ょ", "kleines yo", KanaSubType.SMALL),

    RA("ら", "ra", KanaSubType.BASIC),
    RI("り", "ri", KanaSubType.BASIC),
    RU("る", "ru", KanaSubType.BASIC),
    RE("れ", "re", KanaSubType.BASIC),
    RO("ろ", "ro", KanaSubType.BASIC),

    WA("わ", "wa", KanaSubType.BASIC),
    WO("を", "wo", KanaSubType.BASIC),

    N("ん", "n", KanaSubType.BASIC),

    GA("が", "ga", KanaSubType.WITH_DAKUTEN),
    GI("ぎ", "gi", KanaSubType.WITH_DAKUTEN),
    GU("ぐ", "gu", KanaSubType.WITH_DAKUTEN),
    GE("げ", "ge", KanaSubType.WITH_DAKUTEN),
    GO("ご", "go", KanaSubType.WITH_DAKUTEN),

    ZA("ざ", "za", KanaSubType.WITH_DAKUTEN),
    JI1("じ", "ji", KanaSubType.WITH_DAKUTEN),
    ZU1("ず", "zu", KanaSubType.WITH_DAKUTEN),
    ZE("ぜ", "ze", KanaSubType.WITH_DAKUTEN),
    ZO("ぞ", "zo", KanaSubType.WITH_DAKUTEN),

    DA("だ", "da", KanaSubType.WITH_DAKUTEN),
    JI2("ぢ", "ji", KanaSubType.WITH_DAKUTEN),
    ZU2("づ", "zu", KanaSubType.WITH_DAKUTEN),
    DE("で", "de", KanaSubType.WITH_DAKUTEN),
    DO("ど", "do", KanaSubType.WITH_DAKUTEN),

    BA("ば", "ba", KanaSubType.WITH_DAKUTEN),
    BI("び", "bi", KanaSubType.WITH_DAKUTEN),
    BU("ぶ", "bu", KanaSubType.WITH_DAKUTEN),
    BE("べ", "be", KanaSubType.WITH_DAKUTEN),
    BO("ぼ", "bo", KanaSubType.WITH_DAKUTEN),

    PA("ぱ", "pa", KanaSubType.WITH_HANDAKUTEN),
    PI("ぴ", "pi", KanaSubType.WITH_HANDAKUTEN),
    PU("ぷ", "pu", KanaSubType.WITH_HANDAKUTEN),
    PE("ぺ", "pe", KanaSubType.WITH_HANDAKUTEN),
    PO("ぽ", "po", KanaSubType.WITH_HANDAKUTEN),

    KYA("きゃ", "kya", KanaSubType.COMPOUND),
    KYU("きゅ", "kyu", KanaSubType.COMPOUND),
    KYO("きょ", "kyo", KanaSubType.COMPOUND),

    SHA("しゃ", "sha", KanaSubType.COMPOUND),
    SHU("しゅ", "shu", KanaSubType.COMPOUND),
    SHO("しょ", "sho", KanaSubType.COMPOUND),

    CHA("ちゃ", "cha", KanaSubType.COMPOUND),
    CHU("ちゅ", "chu", KanaSubType.COMPOUND),
    CHO("ちょ", "cho", KanaSubType.COMPOUND),

    NYA("にゃ", "nya", KanaSubType.COMPOUND),
    NYU("にゅ", "nyu", KanaSubType.COMPOUND),
    NYO("にょ", "nyo", KanaSubType.COMPOUND),

    HYA("ひゃ", "hya", KanaSubType.COMPOUND),
    HYU("ひゅ", "hyu", KanaSubType.COMPOUND),
    HYO("ひょ", "hyo", KanaSubType.COMPOUND),

    MYA("みゃ", "mya", KanaSubType.COMPOUND),
    MYU("みゅ", "myu", KanaSubType.COMPOUND),
    MYO("みょ", "myo", KanaSubType.COMPOUND),

    RYA("りゃ", "rya", KanaSubType.COMPOUND),
    RYU("りゅ", "ryu", KanaSubType.COMPOUND),
    RYO("りょ", "ryo", KanaSubType.COMPOUND),

    GYA("ぎゃ", "gya", KanaSubType.COMPOUND_WITH_DAKUTEN),
    GYU("ぎゅ", "gyu", KanaSubType.COMPOUND_WITH_DAKUTEN),
    GYO("ぎょ", "gyo", KanaSubType.COMPOUND_WITH_DAKUTEN),

    JA1("じゃ", "ja", KanaSubType.COMPOUND_WITH_DAKUTEN),
    JU1("じゅ", "ju", KanaSubType.COMPOUND_WITH_DAKUTEN),
    JO1("じょ", "jo", KanaSubType.COMPOUND_WITH_DAKUTEN),

    JA2("ぢゃ", "ja", KanaSubType.COMPOUND_WITH_DAKUTEN),
    JU2("ぢゅ", "ju", KanaSubType.COMPOUND_WITH_DAKUTEN),
    JO2("ぢょ", "jo", KanaSubType.COMPOUND_WITH_DAKUTEN),

    BYA("びゃ", "bya", KanaSubType.COMPOUND_WITH_DAKUTEN),
    BYU("びゅ", "byu", KanaSubType.COMPOUND_WITH_DAKUTEN),
    BYO("びょ", "byo", KanaSubType.COMPOUND_WITH_DAKUTEN),

    PYA("ぴゃ", "pya", KanaSubType.COMPOUND_WITH_HANDAKUTEN),
    PYU("ぴゅ", "pyu", KanaSubType.COMPOUND_WITH_HANDAKUTEN),
    PYO("ぴょ", "pyo", KanaSubType.COMPOUND_WITH_HANDAKUTEN),

    LEHNWORT_WI("ゔぃ", "wi", KanaSubType.FOR_FOREIGN_WORDS),
    LEHNWORT_FA("ふぁ", "fa", KanaSubType.FOR_FOREIGN_WORDS),
    LEHNWORT_TI("てぃ", "ti", KanaSubType.FOR_FOREIGN_WORDS),
    LEHNWORT_DU("どぅ", "du", KanaSubType.FOR_FOREIGN_WORDS),
    LEHNWORT_WE("うぇ", "we", KanaSubType.FOR_FOREIGN_WORDS),
    LEHNWORT_FO("ふぉ", "fo", KanaSubType.FOR_FOREIGN_WORDS),

    SMALL_TSU("っ", "kleines tsu", KanaSubType.SMALL),
    ;

    /** Das Zeichen in UTF-8 Darstellung. */
    private final String character;

    /** Die Hepburn-Darstellung in lateinischen Buchstaben. */
    private final String hepburn;

    /** Die Unterart des Hiragana-Zeichens. */
    private final KanaSubType subType;

    /**
     * Konstruktor.
     *
     * @param character
     *            Das Zeichen in UTF-8 Darstellung.
     * @param hepburn
     *            Die Hepburn-Darstellung in lateinischen Buchstaben.
     * @param subType
     *            Die Unterart des Katakana-Zeichens.
     */
    private Hiragana(String character, String hepburn, KanaSubType subType) {
        this.character = character;
        this.hepburn = hepburn;
        this.subType = subType;
    }

    /** Getter für das Zeichen in UTF-8 Darstellung. */
    public String getCharacter() {
        return character;
    }

    /** Getter für die Hepburn-Darstellung in lateinischen Buchstaben. */
    public String getHepburn() {
        return hepburn;
    }

    /** Getter für die Unterart des Hiragana-Zeichens. */
    public KanaSubType getSubType() {
        return subType;
    }

    /** Gibt das Hiragana-Zeichen als Kana-Objekt zurück. */
    public Kana toKana() {
        Kana kana = new Kana(character, hepburn);
        return kana;
    }

    /** Gibt eine Liste mit den Zeichen der Schrift Hiragana (in UTF-Zeichen) zurück. */
    public static List<String> getAllHiraganaAsStringList() {
        List<String> hiraganaList = new ArrayList<>();

        for (Hiragana hiragana : Hiragana.values()) {
            hiraganaList.add(hiragana.getCharacter());
        }

        return hiraganaList;
    }

    /** Gibt an, ob in dem Text ein Hiragana-Zeichen vorkommt. */
    public static boolean containsHiragana(String text) {
        return Text.contains(text, getAllHiraganaAsStringList());
    }

    /** Gibt an, ob der Text ausschließlich aus Hiragana-Zeichen besteht. */
    public static boolean containsOnlyHiragana(String text) {
        if (text.isEmpty()) {
            return false;
        }
        else {
            List<String> hiragana = getAllHiraganaAsStringList();
            for (String textCharacter : Text.textToCharactersList(text)) {
                if (!hiragana.contains(textCharacter)) {
                    return false;
                }
            }

            return true;
        }
    }

    /**
     * Gibt den Index des ersten Hiragana-Zeichen im Text zurück. Gibt es kein solches, wird -1
     * zurückgegeben.
     */
    public static int firstIndexOfHiragana(String text) {
        List<String> hiragana = getAllHiraganaAsStringList();
        List<String> textCharacters = Text.textToCharactersList(text);

        for (int index = 0; index < textCharacters.size(); ++index) {
            String textCharacter = textCharacters.get(index);
            if (hiragana.contains(textCharacter )) {
                return index;
            }
        }

        return -1;
    }

    /**
     * Gibt den Index des letzten Hiragana-Zeichen im Text zurück. Gibt es kein solches, wird -1
     * zurückgegeben.
     */
    public static int lastIndexOfHiragana(String text) {
        List<String> hiragana = getAllHiraganaAsStringList();
        List<String> textCharacters = Text.textToCharactersList(text);

        for (int index = textCharacters.size() - 1; index >= 0; --index) {
            String textCharacter = textCharacters.get(index);
            if (hiragana.contains(textCharacter )) {
                return index;
            }
        }

        return -1;
    }

    /**
     * Gibt eine Liste mit allen Hiragana zurück, die die gewünschten Unterarten haben.
     *
     * @param kanaSubTypes
     *            Die gewünschten Unterarten.
     * @return Die Liste mit allen Hiragana zurück, die die gewünschten Unterarten haben.
     */
    public static List<Hiragana> getKanasWithSubtypes(KanaSubType ... kanaSubTypes) {
        return getKanasWithSubtypes(CollectionsHelper.arrayToList(kanaSubTypes));
    }

    /**
     * Gibt eine Liste mit allen Hiragana zurück, die die gewünschten Unterarten haben.
     *
     * @param kanaSubTypes
     *            Die gewünschten Unterarten.
     * @return Die Liste mit allen Hiragana zurück, die die gewünschten Unterarten haben.
     */
    public static List<Hiragana> getKanasWithSubtypes(List<KanaSubType> kanaSubTypes) {
        List<Hiragana> list = new ArrayList<>();

        for (Hiragana hiragana : values()) {
            KanaSubType actualSubType = hiragana.getSubType();
            if (kanaSubTypes.contains(actualSubType)) {
                list.add(hiragana);
            }
        }

        return list;
    }

    /** Gibt die Anzahl der bekannten Hiragana zurück. */
    public static int getNumberOfKnownHiragana() {
        return values().length;
    }

}
