package de.duehl.vocabulary.japanese.logic.ownlists;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import de.duehl.vocabulary.japanese.common.data.InternalAdditionalVocableData;
import de.duehl.vocabulary.japanese.data.Vocable;
import de.duehl.vocabulary.japanese.data.Vocabulary;
import de.duehl.vocabulary.japanese.logic.internal.InternalDataRequester;

/**
 * Diese Klasse erstellt ein Verzeichnis der Vokabeln nach ihren Schlüsseln.
 *
 * @version 1.01     2024-01-20
 * @author Christian Dühl
 */

public class KeyToVocableMapCreator {

    /** Die Liste mit den bekannten Vokabularien. */
    private final List<Vocabulary> vocabularies;

    /** Das Objekt das zu einer Vokabel die internen, benutzerabhängigen Daten abrufen kann. */
    private final InternalDataRequester requester;

    /** Das Verzeichnis der Vokabeln nach ihren Schlüsseln. */
    private Map<String, Vocable> keyToVocable;

    /**
     * Konstruktor.
     *
     * @param vocabularies
     *            Die Liste mit den bekannten Vokabularien.
     * @param requester
     *            Das Objekt das zu einer Vokabel die internen, benutzerabhängigen Daten abrufen
     *            kann.
     * @param options
     *            Die Programmoptionen.
     * @param gui
     *            Die grafische Oberfläche.
     */
    public KeyToVocableMapCreator(List<Vocabulary> vocabularies, InternalDataRequester requester) {
        this.vocabularies = vocabularies;
        this.requester = requester;
    }

    /** Erzeugt das Verzeichnis. */
    public void create() {
        keyToVocable = buildKeyToVocableMap(vocabularies);
    }

    private Map<String, Vocable> buildKeyToVocableMap(List<Vocabulary> vocabularies) {
        Map<String, Vocable> keyToVocable = new HashMap<>();

        for (Vocabulary vocabulary : vocabularies) {
            for (Vocable vocable : vocabulary.getVocables()) {
                InternalAdditionalVocableData data = requester.getInternalDataForVocable(vocable);
                String key = data.getKey();
                if (keyToVocable.containsKey(key)) {
                    throw new RuntimeException("Der Schlüssel '" + key + "' kommt mehrfach vor!");
                }
                keyToVocable.put(key, vocable);
            }
        }

        return keyToVocable;
    }

    /** Getter für das Verzeichnis der Vokabeln nach ihren Schlüsseln. */
    public Map<String, Vocable> getKeyToVocable() {
        return keyToVocable;
    }

}
