package de.duehl.vocabulary.japanese.startup.logic;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import de.duehl.basics.io.FileHelper;
import de.duehl.basics.text.NumberString;
import de.duehl.basics.text.Text;
import de.duehl.swing.ui.GuiTools;
import de.duehl.swing.ui.text.TextViewer;
import de.duehl.vocabulary.japanese.common.persistence.Options;
import de.duehl.vocabulary.japanese.data.Vocable;
import de.duehl.vocabulary.japanese.data.Vocabulary;
import de.duehl.vocabulary.japanese.io.VocabularyDirectoryReader;
import de.duehl.vocabulary.japanese.startup.ui.MessageAppender;

/**
 * Diese Klasse liest im Verzeichnis mit den Vokabularien und die Sound-Files (mp3) alle Datei mit
 * Vokabularien ein und trägt in den Vokabeln die Sound-Files (mp3) mit dem korrekten Pfad nach.
 *
 * @version 1.01     2025-03-20
 * @author Christian Dühl
 */

public class VocabularyAndSoundFileFinder {

    private static final boolean INFORM_ABOUT_EACH_VOCABULARY_SOUND_FILE_CREATION = false;
    private static final int MAXIMUM_NON_FATAL_ERRORS_TO_SHOW = 10;
    private static final int MAXIMUM_FATAL_ERRORS_TO_SHOW = 25;


    /** Die Programmoptionen. */
    private final Options options;

    /** Fügt Nachrichten zum Startup hinzu. */
    private final MessageAppender appender;

    /** Das Verzeichnis in dem die Dateien mit den Vokabularien und die Sound-Files (mp3) liegen. */
    private final String directory;

    /** Die Liste der eingelesenen Vokabulare. */
    private List<Vocabulary> vocabularies;

    /**
     * Die Nachricht mit der Anzahl Vokabeln und Vokabularien, die nach dem Laden in der
     * Statuszeile der Oberfläche angezeigt werden soll.
     */
    private String loadUpMessage;

    /** Das Verzeichnis welches den Sound-Dateinamen ohne Pfad solche mit Pfad zuordnet. */
    private Map<String, String> bareFilename2FilenameMap;

    /** Die Liste mit den Meldungen zu doppelten Vokabeln (Kana + Übersetzung). */
    private List<String> doubleVocablesKanaTransltionMessages;

    /** Die Liste mit den Meldungen zu doppelten Vokabeln (Kana + Kanji). */
    private List<String> doubleVocablesKanaKanjiMessages;

    /** Die Liste mit den Meldungen zu Vokabeln mit identischem Kana und Kanji. */
    private List<String> vocablesWithEqualKanaAndKanjiMessages;

    /** Die Liste mit den Meldungen zu fehlenden MP3-Dateinamen ohne Pfad. */
    private List<String> missingMp3BareFilesMessages;

    /** Die Liste mit den Meldungen zu fehlenden MP3-Dateinamen mit Pfad. */
    private List<String> missingMp3FilesMessages;

    /**
     * Die Liste der Fehlermeldungen zu mehrfach vorkommenden Dateinamen ohne Pfad von MP3-Dateien.
     */
    private List<String> doubleMp3BareFilenamesMessages;

    /** Die Liste mit den Zeilen der Meldungen zu fatalen Fehlern. */
    private List<String> fatalErrorLines;

    /**
     * Konstruktor.
     *
     * @param options
     *            Die Optionen.
     */
    public VocabularyAndSoundFileFinder(Options options, MessageAppender appender) {
        this.options = options;
        this.appender = appender;
        directory = options.getVocabulariesPath();

        fatalErrorLines = new ArrayList<>();
        fatalErrorLines.add("Fatale Fehlermeldungen:");
        fatalErrorLines.add("");
    }

    /** Führt die Suche durch. */
    public void find() {
        readVocabularies();
        createAndAppendLoadUpMessage();
        storeVocabularyDescriptionIntoVocable();
        removeVocablesWithoutKanaOrTranslation();
        checkIfSomeVocablesCombinationsAreUnique();
        determineAllSoundFiles();
        adjustMp3InAllVocables();
    }

    private void readVocabularies() {
        VocabularyDirectoryReader reader = new VocabularyDirectoryReader(directory);
        reader.read();
        vocabularies = reader.getVocabularies();
    }

    private void createAndAppendLoadUpMessage() {
        int vocableCount = 0;
        for (Vocabulary vocabulary : vocabularies) {
            vocableCount += vocabulary.getVocables().size();
        }
        loadUpMessage = "Es wurden " + NumberString.taupu(vocabularies.size())
                + " Vokabulare mit zusammen " + NumberString.taupu(vocableCount)
                + " Vokabeln eingelesen.";
        appendMessage(loadUpMessage);

    }

    private void storeVocabularyDescriptionIntoVocable() {
        for (Vocabulary vocabulary : vocabularies) {
            String description = vocabulary.getDescription();
            for (Vocable vocable : vocabulary.getVocables()) {
                vocable.setVocabularyDescription(description);
            }
        }
    }

    private void removeVocablesWithoutKanaOrTranslation() {
        for (Vocabulary vocabulary : vocabularies) {
            removeVocablesWithoutKanaOrTranslation(vocabulary);
        }
    }

    private void removeVocablesWithoutKanaOrTranslation(Vocabulary vocabulary) {
        List<Vocable> vocables = vocabulary.getVocables();

        List<Vocable> vocablesToRemove = new ArrayList<>();
        for (Vocable vocable : vocables) {
            if (shouldBeRemoved(vocable)) {
                vocablesToRemove.add(vocable);
            }
        }

        if (!vocablesToRemove.isEmpty()) {
            appendMessage(vocabulary.getDescription() + ": Die folgenden Variablen werden "
                    + "entfernt, weil sie nicht vollständig genug sind:");
            for (Vocable vocableToRemove : vocablesToRemove) {
                appendMessage(vocableToRemove.toNiceString(4));
                appendMessage("");
                vocables.remove(vocableToRemove);
            }
        }
    }

    private void appendMessage(String message) {
        appender.appendMessage(message);
    }

    private boolean shouldBeRemoved(Vocable vocable) {
        return vocable.getKana().isBlank()
                || vocable.getTranslations().isEmpty()
                || vocable.getTranslations().get(0).isBlank()
                ;
    }

    /**
     * Prüft, ob der Schlüssel aus Kana + erste Übersetzung eindeutig und beides nicht leer ist und
     * ob der die Kombination von Kanji und Kana eindeutig ist.
     */
    private void checkIfSomeVocablesCombinationsAreUnique() {
        doubleVocablesKanaTransltionMessages = new ArrayList<>();
        doubleVocablesKanaKanjiMessages = new ArrayList<>();
        vocablesWithEqualKanaAndKanjiMessages = new ArrayList<>();

        List<Vocable> allVocables = createAllVocablesList();

        for (int index1 = 0; index1 < allVocables.size() - 1; ++index1) {
            Vocable vocable1 = allVocables.get(index1);
            for (int index2 = index1 + 1; index2 < allVocables.size(); ++index2) {
                Vocable vocable2 = allVocables.get(index2);
                checkIfVocablesAreUniqueWithKeyFromKanaAndFirstTranslations(vocable1, vocable2);
                checkIfVocablesAreUniqueWithKanjiAndKana(vocable1, vocable2);
            }
        }

        for (Vocable vocable : allVocables) {
            checkIfVocableHasEqualKanaAndKanji(vocable);
        }

        if (!doubleVocablesKanaTransltionMessages.isEmpty()
                && !doubleVocablesKanaKanjiMessages.isEmpty()) {
            reportAboutFatalErrorsWithoutExit(
                    "Es gibt Vokabeln, welche nach Kana und der ersten Übersetzung gleich sind:",
                    doubleVocablesKanaTransltionMessages);
            reportAboutFatalErrorsAndExit(
                    "Es gibt Vokabeln, welche nach Kana und Kanji gleich sind:",
                    doubleVocablesKanaKanjiMessages);
        }
        if (!doubleVocablesKanaTransltionMessages.isEmpty()) {
            reportAboutFatalErrorsAndExit(
                    "Es gibt Vokabeln, welche nach Kana und der ersten Übersetzung gleich sind:",
                    doubleVocablesKanaTransltionMessages);
        }
        if (!doubleVocablesKanaKanjiMessages.isEmpty()) {
            reportAboutFatalErrorsAndExit(
                    "Es gibt Vokabeln, welche nach Kana und Kanji gleich sind:",
                    doubleVocablesKanaKanjiMessages);
        }
        if (options.isInformAboutEqualKanaAndKanjiAtStartup()
                && !vocablesWithEqualKanaAndKanjiMessages.isEmpty()) {
            reportAboutFatalErrorsAndExit(
                    "Es gibt Vokabeln, welche identische Inhalte von Kana und Kanji haben:",
                    vocablesWithEqualKanaAndKanjiMessages);
        }
    }

    private List<Vocable> createAllVocablesList() {
        List<Vocable> allVocables = new ArrayList<>();

        for (Vocabulary vocabulary : vocabularies) {
            allVocables.addAll(vocabulary.getVocables());
        }

        return allVocables;
    }

    /**
     * Prüft, ob der Schlüssel aus Kana + erste Übersetzung eindeutig und beides nicht leer ist.
     *
     * Wegen removeVocablesWithoutKanaOrTranslation wissen wir, dass beides gefüllt ist und es
     * mindestens eine Übersetzung gibt.
     */
    private void checkIfVocablesAreUniqueWithKeyFromKanaAndFirstTranslations(Vocable vocable1,
            Vocable vocable2) {
        String kana1 = vocable1.getKana();
        String kana2 = vocable2.getKana();

        String translation1 = vocable1.getTranslations().get(0);
        String translation2 = vocable2.getTranslations().get(0);

        if (kana1.equals(kana2) && translation1.equals(translation2)) {
            StringBuilder builder = new StringBuilder();
            builder.append(vocable1.toNiceString(12)).append("\n");
            builder.append("    ").append("und").append("\n");
            builder.append(vocable2.toNiceString(12)).append("\n");
            doubleVocablesKanaTransltionMessages.add(builder.toString());
        }
    }

    /** Prüft, ob der die Kombination von Kanji und Kana eindeutig ist. */
    private void checkIfVocablesAreUniqueWithKanjiAndKana(Vocable vocable1, Vocable vocable2) {
        String kana1 = vocable1.getKana();
        String kana2 = vocable2.getKana();

        String kanji1 = vocable1.getKanji();
        String kanji2 = vocable2.getKanji();

        if (kana1.equals(kana2) && kanji1.equals(kanji2)) {
            StringBuilder builder = new StringBuilder();
            builder.append(vocable1.toNiceString(12)).append("\n");
            builder.append("    ").append("und").append("\n");
            builder.append(vocable2.toNiceString(12)).append("\n");
            doubleVocablesKanaKanjiMessages.add(builder.toString());
        }
    }

    private void checkIfVocableHasEqualKanaAndKanji(Vocable vocable) {
        String kana = vocable.getKana();
        String kanji = vocable.getKanji();

        if (kana.equals(kanji)) {
            StringBuilder builder = new StringBuilder();
            builder.append(vocable.toNiceString(12)).append("\n");
            vocablesWithEqualKanaAndKanjiMessages.add(builder.toString());
        }
    }

    private void determineAllSoundFiles() {
        appendMessage("Suche die Sound-Dateien zu allen Vokabeln ...");
        doubleMp3BareFilenamesMessages = new ArrayList<>();

        List<String> soundFilenames = FileHelper.findFilesNio2(directory, ".mp3");
        bareFilename2FilenameMap = new HashMap<>();
        for (String soundFilename : soundFilenames) {
            String bareSoundFilename = FileHelper.getBareName(soundFilename);
            if (bareFilename2FilenameMap.containsKey(bareSoundFilename)) {
                doubleMp3BareFilenamesMessages.add(""
                        + "    Die MP3-Datei " + bareSoundFilename + " ist zweimal "
                                + "vorhanden:\n"
                        + "        " + "Datei 1: "
                                + bareFilename2FilenameMap.get(bareSoundFilename) + "\n"
                        + "        " + "Datei 2: " + soundFilename + "\n");
            }
            bareFilename2FilenameMap.put(bareSoundFilename, soundFilename);
        }

        if (!doubleMp3BareFilenamesMessages.isEmpty()
                && options.isInformAboutDoubleMp3AtStartup()) {
            reportDoubleMp3BareFilenames();
        }
    }

    /**
     * Da gibt es tatsächlich Fälle wo das in Ordnung ist, etwa bei shi:
     *     1) 四 (し, shi) = vier, 四 ergibt aber den Klang "yon".
     *     2) 市 (し, shi) = Stadt, 市 ergibt aber den Klang "ichi".
     * Oder bei ka:
     *     1) 火 (か, ka) = Dienstag (Kurzform von 火曜日 (かようび, kayoubi)
     *     2) か als Question Marker
     * Oder bei sh(i)ta:
     *     1) 下 (した, shita) = unten (mit Kanji kommt da ein falscher Klang)
     *     2) した = getan haben (Ta-Form von 「為る」 (「する」, suru))
     *
     * In anderen Fällen können es aber echte Dubletten sein, deren erste Übersetzung sich
     * unterscheidet.
     *
     * Neuerdings sehe ich das als fatalen Fehler und möchte, dass man je eine der Dateien
     * entfernt!
     */
    private void reportDoubleMp3BareFilenames() {
        reportAboutFatalErrorsAndExit(
                "Achtung, es kommen Dateinamen (ohne Pfad) von MP3-Dateien mehrfach vor:",
                doubleMp3BareFilenamesMessages);
        /*
         * Wird neuerdings als fataler Fehler gesehen!
        appendMessage("Es gibt Fälle, wo das in Ordnung ist, z.B.\n"
                + "    - 四 / 市 (し, shi)\n"
                + "    - 火 (か, ka) / か als question marker und Kurzform von Dienstag\n"
                + "    - 下 (した, shita) \"unten\" / した \"getan haben\"\n"
                + "andere Fälle können auf Fehler hinweisen.\n");
                */
    }

    private void adjustMp3InAllVocables() {
        appendMessage("Trage die gefundenen Sound-Dateien in allen Vokabeln ein ...");
        missingMp3BareFilesMessages = new ArrayList<>();
        missingMp3FilesMessages = new ArrayList<>();

        for (Vocabulary vocabulary : vocabularies) {
            if (INFORM_ABOUT_EACH_VOCABULARY_SOUND_FILE_CREATION) {
                appendMessage("Ergänze Sound-Files für " + vocabulary.getDescription());
            }
            for (Vocable vocable : vocabulary.getVocables()) {
                searchAndStoreSoundFileWithPath(vocable);
            }
        }

        if (!missingMp3BareFilesMessages.isEmpty()) {
            reportMissingMp3BareFilesAndExit();
        }
        if (!missingMp3FilesMessages.isEmpty()) {
            reportMissingMp3FilesAndExit();
        }
    }

    private void searchAndStoreSoundFileWithPath(Vocable vocable) {
        String bareMp3 = vocable.getBareMp3();

        if (bareMp3.isBlank()) {
            missingMp3BareFilesMessages.add(""
                    + "    Die Sound-Datei ohne Pfad '" + bareMp3 + "' ist leer.\n"
                    + "        " + "vocable = " + vocable + "\n");
        }
        else {
            if (bareFilename2FilenameMap.containsKey(bareMp3)) {
                String filename = bareFilename2FilenameMap.get(bareMp3);
                vocable.setMp3(filename);
            }
            else {
                missingMp3FilesMessages.add(""
                        + "    Die Sound-Datei '" + bareMp3 + "' wurde nicht gefunden.\n"
                        + "        " + "vocable = " + vocable + "\n");
            }
        }
    }

    private void reportMissingMp3BareFilesAndExit() {
        reportAboutFatalErrorsAndExit(
                "Es gibt Vokabeln ohne eingetragenen Namen der MP3-Datei (ohne Pfad):",
                missingMp3BareFilesMessages);
    }

    private void reportMissingMp3FilesAndExit() {
        int numberOfMissingMp3Files = missingMp3FilesMessages.size();
        String title = "Es gibt " + numberOfMissingMp3Files
                + " Vokabeln zu denen keine MP3-Datei (mit Pfad) gefunden wurde";
        String description = title + ":";

        if (options.isAllowMissingMp3()) {
            if (options.isReportMissingMp3()) {
                reportAboutNotFatalErrors(description, missingMp3FilesMessages);
            }
        }
        else {
            StringBuilder builder = new StringBuilder();
            builder.append(title).append(".\n\n");
            int max = Math.min(3, numberOfMissingMp3Files);
            boolean more = max < numberOfMissingMp3Files;
            for (int index = 0; index < max; ++index) {
                if (max > 1) {
                    builder.append((index + 1) + ")");
                }
                builder.append(missingMp3FilesMessages.get(index)).append("\n");
            }
            if (more) {
                builder.append("...\n");
            }
            builder.append("\nSoll trrotzdem gestartet werden?\n"
                    + "Dann wird auch die Option, dass trotz fehlender MP3-Dateien gestartet "
                    + "wird, gesetzt.");
            String message = builder.toString();
            boolean goOn = GuiTools.askUser(title, message);
            if (goOn) {
                options.setAllowMissingMp3(true);
            }
            else {
                /* Weil es nun zu viele Vokabeln sind, spare ich mir das:
                appendMessage("Übersicht der bekannten MP3-Namen:");
                for (String bareFilenameInMap :
                        CollectionsHelper.getSortedMapStringIndices(bareFilename2FilenameMap)) {
                    appendMessage(bareFilenameInMap + " -> "
                            + bareFilename2FilenameMap.get(bareFilenameInMap));
                }
                */
                reportAboutFatalErrorsAndExit(description, missingMp3FilesMessages);
            }
        }
    }

    private void reportAboutNotFatalErrors(String description, List<String> errorMessages) {
        appendMessage(description);
        int numberOfErrors = errorMessages.size();
        boolean moreErrors = numberOfErrors > MAXIMUM_NON_FATAL_ERRORS_TO_SHOW;
        int numberOfErrorsToShow = Math.min(numberOfErrors, MAXIMUM_NON_FATAL_ERRORS_TO_SHOW);
        for (int index = 0; index < numberOfErrorsToShow; ++index) {
            if (errorMessages.size() > 1) {
                appendMessage((index + 1) + ") ");
            }
            appendMessage(errorMessages.get(index));
        }
        if (moreErrors) {
            appendMessage("... und " + NumberString.taupu(numberOfErrors - numberOfErrorsToShow)
                    + " weitere Fehler dieser Art.");
        }
    }

    private void reportAboutFatalErrorsAndExit(String description, List<String> errorMessages) {
        reportAboutFatalErrorsWithoutExit(description, errorMessages);
        fatalErrorLines.add("Daher wird der Start abgebrochen.");
        printFatalErrorMessagesToSytemErr();
        showFatalErrorMessagesAsDialog();
        System.exit(1);
    }

    private void reportAboutFatalErrorsWithoutExit(String description, List<String> errorMessages) {
        fatalErrorLines.add(description);
        int numberOfErrors = errorMessages.size();
        boolean moreErrors = numberOfErrors > MAXIMUM_FATAL_ERRORS_TO_SHOW;
        int numberOfErrorsToShow = Math.min(numberOfErrors, MAXIMUM_FATAL_ERRORS_TO_SHOW);
        for (int index = 0; index < numberOfErrorsToShow; ++index) {
            if (errorMessages.size() > 1) {
                fatalErrorLines.add((index + 1) + ") ");
            }
            fatalErrorLines.add(errorMessages.get(index));
        }
        if (moreErrors) {
            fatalErrorLines.add("... und "
                    + NumberString.taupu(numberOfErrors - numberOfErrorsToShow)
                    + " weitere Fehler dieser Art.");
        }
    }

    private void printFatalErrorMessagesToSytemErr() {
        for (String line : fatalErrorLines) {
            System.err.println(line);
        }
    }

    private void showFatalErrorMessagesAsDialog() {
        TextViewer viewer = new TextViewer("Fatale Fehlermedlungen");
        viewer.setText(Text.joinWithLineBreak(fatalErrorLines));
        viewer.setVisible(true);
    }

    /** Getter für die Liste der eingelesenen Vokabulare. */
    public List<Vocabulary> getVocabularies() {
        return vocabularies;
    }

    /**
     * Getter für die Nachricht mit der Anzahl Vokabeln und Vokabularien, die nach dem Laden in der
     * Statuszeile der Oberfläche angezeigt werden soll.
     */
    public String getLoadUpMessage() {
        return loadUpMessage;
    }

}
