package de.duehl.vocabulary.japanese.ui.components.text;

import de.duehl.basics.text.Text;
import de.duehl.vocabulary.japanese.data.Vocable;

/**
 * Diese Klasse erstellt den Text zu Anzeige für Romaji und Aussprache einer Vokabel.
 *
 * @version 1.01     2024-03-12
 * @author Christian Dühl
 */

public class RomajiAndPronunciationTextCreator {

    private static final int CHARACTERS_PER_LINE = 35;


    /** Die Vokabel, welche angezeigt wird. */
    private final Vocable vocable;

    /** Gibt an, ob die Romaji angezeigt werden sollen. */
    private final boolean showRomaji;

    /** Gibt an, ob die Aussprache angezeigt werden soll. */
    private final boolean showPronunciation;

    /**
     * Der Haupt-Text, enthält die Romaji, wenn dies angezeigt werden soll. Kann leer sein, wenn er
     * nicht angezeigt werden soll.
     */
    private String firstTextPart;

    /**
     * Der Text, der in Klammern angezeigt werden soll, also normalerweise die Aussprache. Wenn man
     * den nicht braucht, ist er leer. Man muss ihn noch mit Klammern umgeben, wenn er nicht leer
     * ist.
     */
    private String textPartInBrace;

    /** Gibt an, ob die Klammer in einer eigenen Zeile dargestellt werden soll. */
    private boolean braceInOwnLine;

    /**
     * Konstruktor.
     *
     * @param vocable
     *            Die Vokabel, welche angezeigt wird.
     *            @param showRomaji Gibt an, ob die Romaji angezeigt werden sollen.
     *            @param showPronunciation Gibt an, ob die Aussprache angezeigt werden soll.
     */
    public RomajiAndPronunciationTextCreator(Vocable vocable, boolean showRomaji,
            boolean showPronunciation) {
        this.vocable = vocable;
        this.showRomaji = showRomaji;
        this.showPronunciation = showPronunciation;
    }

    /** Erzeugt den Text für die Anzeige von Kanji und Kana. */
    public void create() {
        initializeTexts();
        breakLongTexts();
    }

    private void initializeTexts() {
        firstTextPart = vocable.getRomaji();
        textPartInBrace = vocable.getPronunciation();

        if (!showPronunciation) {
            textPartInBrace = "";
        }
        if (!showRomaji) {
            firstTextPart = textPartInBrace;
            textPartInBrace = "";
        }

        braceInOwnLine = !textPartInBrace.isEmpty()
                && firstTextPart.length() + textPartInBrace.length() + 5 > CHARACTERS_PER_LINE;
                // 5: Klammern, Anführungzeichen, Leerzeichen
    }

    private void breakLongTexts() {
        firstTextPart = breakLongText(firstTextPart);
        textPartInBrace = breakLongText(textPartInBrace);
    }

    private String breakLongText(String text) {
        if (text.length() > CHARACTERS_PER_LINE) {
            return Text.addLineBreaks(text, CHARACTERS_PER_LINE);
        }
        else {
            return text;
        }
    }

    /** Getter für den Haupt-Text, enthält Kanji oder Kana, falls kein Kanji gesetzt ist. */
    public String getFirstTextPart() {
        return firstTextPart;
    }

    /**
     * Getter für den Text, der in Klammern angezeigt werden soll, also normalerweise Kana. Wenn
     * man den nicht braucht, ist er leer. Man muss ihn noch mit Klammern umgeben, wenn er nicht
     * leer ist.
     */
    public String getTextPartInBrace() {
        return textPartInBrace;
    }

    /** Gibt an, ob die Klammer in einer eigenen Zeile dargestellt werden soll. */
    public boolean isBraceInOwnLine() {
        return braceInOwnLine;
    }

}
