package de.duehl.vocabulary.japanese.ui.dialog.detail.addtolist;

import java.awt.Image;
import java.awt.Point;
import java.util.List;

import de.duehl.swing.ui.GuiTools;
import de.duehl.vocabulary.japanese.common.persistence.Options;
import de.duehl.vocabulary.japanese.common.persistence.data.HistoricalOwnListPersistanceDataList;
import de.duehl.vocabulary.japanese.data.OwnList;
import de.duehl.vocabulary.japanese.data.Vocable;
import de.duehl.vocabulary.japanese.logic.ownlists.OwnLists;
import de.duehl.vocabulary.japanese.ui.dialog.ownlist.OwnListSelectionDialog;

/**
 * Diese Klasse fügt eine Vokabel zu einer eigenen Liste hinzu.
 *
 * @version 1.01     2025-07-14
 * @author Christian Dühl
 */

class VocableToOwnListAdder {

    /** Die Verwaltung der eigenen Vokabellisten. */
    private final OwnLists ownLists;

    /**
     * Die Position des Rahmens der Oberfläche, die den Panel anzeigt, welcher diese Klasse
     * verwendet.
     */
    private final Point location;

    /** Das Icon für das Programm. */
    private final Image programImage;

    /**
     * Konstruktor.
     *
     * @param ownLists
     *            Die Verwaltung der eigenen Vokabellisten.
     * @param location
     *            Die Position des Rahmens der Oberfläche, die den Panel anzeigt, welcher diese
     *            Klasse verwendet.
     * @param programImage
     *            Das Icon für das Programm.
     */
    public VocableToOwnListAdder(OwnLists ownLists, Point location, Image programImage) {
        this.ownLists = ownLists;
        this.location = location;
        this.programImage = programImage;
    }

    /**
     * Fügt die angezeigte Vokabel zu einer eigenen Liste hinzu.
     *
     * Dazu wählt der Benutzer eine eigene Liste aus. Wenn der Dialog mit dem Button "Angezeigte
     * Liste auswählen" beendet wurde und die gewählte Liste die Vokabel noch nicht enthält, wird
     * sie hinzugefügt.
     *
     * In diesem Erfolgsfall muss man die Listen speichern, da ja eine Vokabel hinzugefügt wurde.
     * Daher wird die Methode
     *     ownLists.storeOwnLists();
     * aufgerufen.
     * Man muss zwar nicht in der Gui die Listen neu aufbauen, denn an deren Anzahl und Namen wurde
     * nichts geändert, ebensowenig an der Anordnung der Listen etc., aber durch die geänderte
     * Füllung können sich die Farben und Prozentzahlen der Listen-Bars ändern. Daher könnte man
     * denken dass es reicht, die Methode
     *     ownLists.setCorrectForegroundColorOfVocabularyBarsInGui();
     * aufzurufen, durch welche die Farben und Prozente aktualisiert werden. Allerdings werden dann
     * bei einer vorher leeren Liste die Buttons nicht farbig und aktiv. Daher ruft man die
     * Methode
     *     ownLists.actualizeOwnListsPartInGui();
     * auf
     */
    public void addVocableToList(Vocable vocable) {
        String title = "Auswahl einer eigenen Liste";
        String text = "Bitte wählen Sie die eigene Liste aus, zu der die Vokabel "
                + vocable.getKanjiKanaRomajiWithJapaneseBraces() + " hinzugefügt werden soll.";
        OwnListSelectionDialog dialog = new OwnListSelectionDialog(title, text, ownLists, location,
                programImage);
        dialog.setVisible(true);
        if (dialog.isApplied()) {
            OwnList ownList = dialog.getOwnList();
            Options options = ownLists.getOptions();
            HistoricalOwnListPersistanceDataList historicalOwnLists =
                    options.getHistoricalOwnLists();
            historicalOwnLists.addOwnListData(ownList.toOwnListPersistanceData());

            List<Vocable> vocables = ownList.getVocables();
            if (vocables.contains(vocable)) {
                GuiTools.informUser("Die Vokabel ist in der Liste schon enthalten",
                        "Die Vokabel " + vocable.getKanjiKanaRomajiWithJapaneseBraces()
                            + " ist in der Liste '" + ownList.getName() + "' schon enthalten.");
            }
            else {
                vocables.add(vocable);
                GuiTools.informUser("Die Vokabel wurde zur Liste hinzugefügt",
                        "Die Vokabel " + vocable.getKanjiKanaRomajiWithJapaneseBraces()
                            + " wurde zur Liste '" + ownList.getName() + "' hinzugefügt.");
                ownLists.storeOwnLists();
                ownLists.actualizeOwnListsPartInGui();
            }
        }
    }

}
