package de.duehl.vocabulary.japanese.ui.dialog.options;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Image;
import java.awt.Point;
import java.util.ArrayList;
import java.util.List;

import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JPanel;
import javax.swing.JRadioButton;

import de.duehl.basics.collections.CollectionsHelper;
import de.duehl.basics.text.Text;
import de.duehl.basics.text.html.HtmlTool;
import de.duehl.swing.ui.GuiTools;
import de.duehl.swing.ui.components.selections.DirSelection;
import de.duehl.swing.ui.dialogs.base.ModalDialogBase;
import de.duehl.swing.ui.elements.sidelist.SideListOrganiserPanel;
import de.duehl.swing.ui.elements.size.ChangeButtonWithSizeChooser;
import de.duehl.swing.ui.elements.size.SizeChooser;
import de.duehl.swing.ui.layout.VerticalLayout;
import de.duehl.vocabulary.japanese.VocabularyTrainerVersion;
import de.duehl.vocabulary.japanese.common.color.VocableColors;
import de.duehl.vocabulary.japanese.common.data.VocablesShuffleType;
import de.duehl.vocabulary.japanese.common.data.VocablesSubsetSelectionType;
import de.duehl.vocabulary.japanese.common.data.VocabularySortOrder;
import de.duehl.vocabulary.japanese.common.persistence.Options;
import de.duehl.vocabulary.japanese.common.persistence.data.EmptyTranslationsAcceptance;
import de.duehl.vocabulary.japanese.ui.dialog.options.colors.eleven.ElevenColorsOptionsUi;
import de.duehl.vocabulary.japanese.ui.dialog.options.colors.vocable.VocableColorsOptionsUi;

/**
 * Diese Klasse erzeugt die grafische Oberfläche des Optionendialoges des Vokabel-Trainers.
 *
 * Folgende Optionen werden nicht hier im Optionendialog bearbeitet:
 *     stringSelectionFontSize
 *     stringSelectionLabelFontSize
 *     lastUsedVersion
 *     lastShownMainTabIndex
 *     lastShownVocabularyCategoryTabIndex
 *     lastShownOwnListCategoryTabIndex
 *     lastShownVocabularySubCategoryTabIndex
 *     lastShownOwnListSubCategoryTabIndex
 *     lastUsedOwnListImportExportDirectory
 *     numberOfDatasetsPerPageOfOwnListEditor
 *     numberOfColumnsOfOwnListEditor
 *     locationOfOwnListEditor
 *     sizeOfOwnListEditor
 *     translationDirection
 *     lastUsed7ZipDirectory
 *     lastSelectedKanjiSetTextInCombobox
 *     historicalLeftOwnListsInListCommander
 *     historicalRightOwnListsInListCommander
 *     historicalUsedOwnLists
 *
 * @version 1.01     2025-09-14
 * @author Christian Dühl
 */

public class OptionsDialog extends ModalDialogBase {

    private static final Dimension DILOG_DIMENSION = new Dimension(1000, 800);


    /** Programmoptionen. */
    private final Options options;

    /* Gui-Elements: */
    private final DirSelection vocabulariesPathSelection;
    private final JCheckBox switchToNextVocableAfterEntringTranslationCheckBox;
    private final JCheckBox showKanaWhenTestingVocableCheckBox;
    private final JCheckBox showKanjiWhenTestingVocableCheckBox;
    private final JCheckBox hideKanjiWhenTestingVocableAndKanaContainsOnlyHiraganaCheckBox;
    private final JCheckBox showRomajiWhenTestingVocableCheckBox;
    private final JCheckBox hideRomajiWhenTestingVocableAndKanaContainsOnlyHiraganaCheckBox;
    private final JCheckBox showPronunciationWhenTestingVocableCheckBox;
    private final JCheckBox hidePronunciationWhenTestingVocableAndKanaContainsOnlyHiraganaCheckBox;
    private final JCheckBox showCommentWhenTestingVocableCheckBox;
    private final JCheckBox showVocabularyDescriptionWhenTestingVocableCheckBox;
    private final JCheckBox colorVocableDependingOnLastSuccessWhenTestingVocableCheckBox;
    private final JCheckBox colorVocableDependingOnLastSuccessWhenViewingVocableCheckBox;
    private final JCheckBox colorVocabularyDependingOnLastSuccessCheckBox;
    private final JCheckBox hideStartOfVocabularyDescriptionCheckBox;
    private final JCheckBox useTestOrderForViewingVocablesCheckBox;
    private final JCheckBox useOnlyFirstGermanTermTestingCheckBox;

    private final List<VocabularySortOrder> vocabularySortOrders;
    private final List<JRadioButton> vocabularySortOrderRadioButtons;

    private final List<VocablesShuffleType> vocablesShuffleTypes;
    private final List<JRadioButton> vocablesShuffleTypesRadioButtons;
    private final JCheckBox reverseShuffledVocablesCheckBox;
    private final JCheckBox useVocablesShuffleTypeForShowingListsAndSheetsTooCheckBox;

    private final List<VocablesSubsetSelectionType> vocablesSubsetSelectionTypes;
    private final List<JRadioButton> vocablesSubsetSelectionTypesRadioButtons;

    private final JCheckBox showSuccessPercentInVocabularyBarCheckBox;
    private final ChangeButtonWithSizeChooser numberOfVocabularyBarColumnsChooser;

    private final JCheckBox hideVocablesFromListInSearchAreaInOwnListEditorCheckBox;

    private final ChangeButtonWithSizeChooser numberOfColumnsOfBlattDarstellungChooser;
    private final JCheckBox allowMissingMp3CheckBox;
    private final JCheckBox reportMissingMp3CheckBox;
    private final JCheckBox informAboutDoubleMp3AtStartupCheckBox;
    private final JCheckBox informAboutEqualKanaAndKanjiAtStartupCheckBox;

    private final JCheckBox showDetailsButtonInEinzeldarstellungCheckBox;
    private final JCheckBox showDetailsButtonInVokabelblattDarstellungCheckBox;
    private final JCheckBox hideLessStartOfVocabularyDescriptionForVhsCheckBox;

    private final JCheckBox checkNewOwnListsVersionAtStartCheckBox;

    private final JCheckBox showKanaInKanjiSizeWhenTestingVocableCheckBox;

    private final List<EmptyTranslationsAcceptance> emptyTranslationsAcceptances;
    private final List<JRadioButton> emptyTranslationsAcceptancesRadioButtons;

    private final JButton numberOfDaysForAWhileButton;
    private final JButton numberOfDaysAVocableIsNewButton;

    private int numberOfDaysForAWhile;
    private int numberOfDaysAVocableIsNew;

    private final ElevenColorsOptionsUi elevenColorsUi;
    private final VocableColorsOptionsUi vocablesColorsUi;

    private final JCheckBox useSplitPaneInOwnListEditorCheckBox;
    private final JCheckBox showTimestampsInStartupCheckBox;
    private final JCheckBox createGermanJapaneseTranslationAtStartupCheckBox;

    private final JCheckBox informAboutEOrUInVocableFieldAusspracheAtStartupCheckBox;

    private int numberOfRememberedOwnLists;
    private final JButton numberOfRememberedOwnListsButton;

    private final JCheckBox showAllTranslationsCheckBox;
    private int numberOfShownTranslations;
    private final JButton numberOfShownTranslationsButton;

    private final JCheckBox showNumberOfListsInListerCheckBox;

    private final JCheckBox showDetailBottomButonsInEinzeldarstellungTooCheckBox;
    private final JCheckBox showDetailBottomButonsInTranslationEvaluationTooCheckBox;

    private final JCheckBox playMP3WhenTestingVocablesCheckBox;
    private final JCheckBox colorUserInputAtKanjiTestEvaluationDialogCheckBox;

    /**
     * Gibt an, ob nur die Optionen für den Dialog zur Bearbeitung eigener Listen angezeigt werden.
     */
    private boolean showOnlyOwnListDialogOptions;

    /** Gibt an, ob der Dialog mit Klick auf OK beendet wurde. */
    private boolean applied;

    /** Das Organisationselement mit einer Seitenliste und zugehörigen Anzeigelementen .*/
    private final SideListOrganiserPanel sideListOrganiser;

    /**
     * Konstruktor.
     *
     * @param options
     *            Optionen des Programms.
     * @param parentLocation
     *            Position des Rahmens der Oberfläche, vor der dieser Dialog erzeugt wird.
     * @param programImage
     *            Anzuzeigendes ProgrammIcon.
     */
    public OptionsDialog(Options options, Point parentLocation, Image programImage) {
        super(parentLocation, programImage, "Optionen", DILOG_DIMENSION);
        this.options = options;

        vocabulariesPathSelection =
                new DirSelection("Der Pfad zu den Vokabularien und den Sound-Files:");
        switchToNextVocableAfterEntringTranslationCheckBox =
                new JCheckBox(breakLongText("Nach der Eingabe der Übersetzung wird gleich zur "
                        + "nächsten Vokabel gesprungen."));
        showKanaWhenTestingVocableCheckBox =
                new JCheckBox(breakLongText("Beim Abfragen der Vokabeln wird die Darstellung in "
                        + "Kana mit angezeigt."));
        showKanjiWhenTestingVocableCheckBox =
                new JCheckBox(breakLongText("Beim Abfragen der Vokabeln wird die Darstellung in "
                        + "Kanji mit angezeigt."));
        hideKanjiWhenTestingVocableAndKanaContainsOnlyHiraganaCheckBox =
                new JCheckBox(breakLongText("Beim Abfragen von reinen Hiragana-Vokabeln wird die "
                        + "Darstellung in Kanji versteckt."));
        showRomajiWhenTestingVocableCheckBox =
                new JCheckBox(breakLongText("Beim Abfragen der Vokabeln wird die Darstellung in "
                        + "Romaji mit angezeigt."));
        hideRomajiWhenTestingVocableAndKanaContainsOnlyHiraganaCheckBox =
                new JCheckBox(breakLongText("Beim Abfragen von reinen Hiragana-Vokabeln wird die "
                        + "Darstellung in Romaji versteckt."));
        showPronunciationWhenTestingVocableCheckBox =
                new JCheckBox(breakLongText("Beim Abfragen der Vokabeln wird die Aussprache mit "
                        + "angezeigt."));
        hidePronunciationWhenTestingVocableAndKanaContainsOnlyHiraganaCheckBox =
                new JCheckBox(breakLongText("Beim Abfragen von reinen Hiragana-Vokabeln wird die "
                        + "Aussprache versteckt."));
        showCommentWhenTestingVocableCheckBox =
                new JCheckBox(breakLongText("Beim Abfragen der Vokabeln wird der Kommentar mit "
                        + "angezeigt."));
        showVocabularyDescriptionWhenTestingVocableCheckBox =
                new JCheckBox(breakLongText("Beim Abfragen der Vokabeln wird die Beschreibung des "
                        + "Vokabulars mit angezeigt."));
        colorVocableDependingOnLastSuccessWhenTestingVocableCheckBox =
                new JCheckBox(breakLongText("Beim Abfragen der Vokabeln wird die Darstellung "
                        + "abhängig vom Erfolg bei den letzten zehn Abfragen eingefärbt."));
        colorVocableDependingOnLastSuccessWhenViewingVocableCheckBox =
                new JCheckBox(breakLongText("Beim Anzeigen der Vokabeln wird die Darstellung "
                        + "abhängig vom Erfolg bei den letzten zehn Abfragen eingefärbt."));
        colorVocabularyDependingOnLastSuccessCheckBox =
                new JCheckBox(breakLongText("Die Darstellung der angezeigten Vokabularien wird "
                        + "abhängig vom Erfolg bei den letzten zehn Abfragen eingefärbt."));
        hideStartOfVocabularyDescriptionCheckBox = new JCheckBox(breakLongText("Der Anfang der "
                + "Beschreibung der Vokabularien (\"B1_K4_1_\" oder \"X_\") wird ausgeblendet."));
        useTestOrderForViewingVocablesCheckBox = new JCheckBox(breakLongText("Zum Betrachten der "
                + "Vokabeln wird die Reihenfolge zum Abfragen verwendet."));
        allowMissingMp3CheckBox = new JCheckBox(breakLongText("Das Programm wird auch trotz "
                + "fehlender MP3-Dateien gestartet."));
        reportMissingMp3CheckBox = new JCheckBox(breakLongText("Das Programm weist beim Start auf "
                + "fehlende MP3-Dateien hin."));
        informAboutDoubleMp3AtStartupCheckBox = new JCheckBox(breakLongText("Beim Start des "
                + "Programms soll über doppelte MP3-Dateien mit dem gleichen Namen informiert "
                + "werden."));
        informAboutEqualKanaAndKanjiAtStartupCheckBox = new JCheckBox(breakLongText("Beim Start "
                + "des Programms soll über gleiche Kana und Kanji informiert werden."));

        useOnlyFirstGermanTermTestingCheckBox = new JCheckBox(breakLongText("Bei der Übersetzung "
                + "von Deutsch nach Japanisch wird nur der erste (statt einem zufälligen) deutsche "
                + "Begriff, welcher in der Vokabel hinterlegt ist, abgefragt."));

        vocabularySortOrders = new ArrayList<>();
        vocabularySortOrderRadioButtons = new ArrayList<>();

        vocablesShuffleTypes = new ArrayList<>();
        vocablesShuffleTypesRadioButtons = new ArrayList<>();
        reverseShuffledVocablesCheckBox = new JCheckBox(breakLongText("Die unten festgelegte "
                + "Sortierung von Vokabellisten wird umgedreht."));
        useVocablesShuffleTypeForShowingListsAndSheetsTooCheckBox = new JCheckBox(breakLongText(
                "Die Sortierung von Vokabellisten wird auch zum Anzeigen der Listen oder Blätter "
                        + "verwendet."));

        vocablesSubsetSelectionTypes = new ArrayList<>();
        vocablesSubsetSelectionTypesRadioButtons = new ArrayList<>();

        showSuccessPercentInVocabularyBarCheckBox = new JCheckBox(breakLongText("Der Erfolg bei "
                + "den letzten zehn Abfragen der Vokabeln eines Vokabulars wird als Prozentzahl "
                + "mit angezeigt."));

        numberOfVocabularyBarColumnsChooser = createNumberOfVocabularyBarColumnsChooser();
        numberOfColumnsOfBlattDarstellungChooser = createNumberOfColumnsOfBlattDarstellungChooser();

        hideVocablesFromListInSearchAreaInOwnListEditorCheckBox = new JCheckBox(breakLongText(
                "Vokabeln aus der Liste rechts im Suchbereich werden versteckt. Falls nicht, "
                            + "werden sie nur anders dargestellt."));

        showDetailsButtonInEinzeldarstellungCheckBox = new JCheckBox(breakLongText("Der Button "
                + "zum Anzeigen der Details wird in der Einzeldarstellung angezeigt."));
        showDetailsButtonInVokabelblattDarstellungCheckBox = new JCheckBox(breakLongText("Der "
                + "Button zum Anzeigen der Details wird in der Darstellung als Vokabelblatt "
                + "angezeigt."));
        hideLessStartOfVocabularyDescriptionForVhsCheckBox = new JCheckBox(breakLongText("Bei "
                + "Vokabularien und Listen aus der VHS wird weniger vom Anfang der Beschreibung "
                + "der Vokabularien ausgeblendet"));

        checkNewOwnListsVersionAtStartCheckBox = new JCheckBox(breakLongText(
                "Auf neuere Version der eigenen Listen prüfen."));

        showKanaInKanjiSizeWhenTestingVocableCheckBox = new JCheckBox(breakLongText("Stellt die "
                + "Kana, wenn auch Kanji dargestellt werden, genauso groß wie die Kanji dar."));

        emptyTranslationsAcceptances = new ArrayList<>();
        emptyTranslationsAcceptancesRadioButtons = new ArrayList<>();

        sideListOrganiser = new SideListOrganiserPanel();

        numberOfDaysForAWhileButton = new JButton();
        numberOfDaysAVocableIsNewButton = new JButton();

        elevenColorsUi = new ElevenColorsOptionsUi(new VocableColors(options));
        vocablesColorsUi = new VocableColorsOptionsUi();

        useSplitPaneInOwnListEditorCheckBox = new JCheckBox(breakLongText("Einen variablen Trenner "
                + "zwischen dem linken und rechten Teil verwenden."));
        showTimestampsInStartupCheckBox = new JCheckBox(breakLongText("Beim Startup wird die bis "
                + "zu diesem Schritt bereits benötigte Laufzeit vor den Meldungen ausgegeben "
                + "(das ist natürlich erst beim nächsten Start des Programms wirksam)."));
        createGermanJapaneseTranslationAtStartupCheckBox = new JCheckBox(breakLongText("Das "
                + "Verzeichnis für die Übersetzung Deutsch nach Japanisch wird beim Start "
                + "aufgebaut. Dies dauert einige Zeit, die der Start länger dauert. Im anderen "
                + "Fall wird dieses Verzeichnis erst aufgebaut, wenn es das erste Mal benötigt "
                + "wird, also wenn die erste Vokabel von Deutsch nach Japanisch übersetzt wird. "
                + "Übersetzt man oft auch von Deutsch nach Japanisch, sollte man es beim Startup "
                + "erzeugen, anderenfalls ist es besser, es erst dann zu erzeugen, wenn man es "
                + "braucht, weil man es meistens nicht braucht und so der Vokabel-Trainer "
                + "schneller startet."));

        informAboutEOrUInVocableFieldAusspracheAtStartupCheckBox = new JCheckBox(breakLongText(
                "Beim Start wird geprüft, ob \"e\" oder \"u\" in der Aussprache der Vokabeln "
                        + "vorkommt."));

        numberOfRememberedOwnListsButton = new JButton();

        showAllTranslationsCheckBox = new JCheckBox(breakLongText("Alle Übersetzungen anzeigen"));
        numberOfShownTranslationsButton = new JButton();

        showNumberOfListsInListerCheckBox = new JCheckBox(breakLongText("Die Anzahl der Listen, in "
                + "der die Vokabel ist, wird zusammen mit einem Hinweis auf Lernlisten in der "
                + "Listendarstellung eines Vokabulars oder einer eigenen Liste angezeigt."));
        showDetailBottomButonsInEinzeldarstellungTooCheckBox = new JCheckBox(breakLongText(""
                + "Die Buttons zum Anzeigen der eigenen Listen, welche die Vokabel enthalten, zum "
                + "Hinzufügen der Vokabel zu einer eigenen Liste und ggf. dem Button zum Anzeigen "
                + "der bekannten Verb- bzw. Adjektivformen, werden auch im Dialog mit der "
                + "Einzeldarstellung angezeigt."));
        showDetailBottomButonsInTranslationEvaluationTooCheckBox = new JCheckBox(breakLongText(""
                + "Die Buttons zum Anzeigen der eigenen Listen, welche die Vokabel enthalten, zum "
                + "Hinzufügen der Vokabel zu einer eigenen Liste und ggf. dem Button zum Anzeigen "
                + "der bekannten Verb- bzw. Adjektivformen, werden auch im Dialog mit der Bewertung "
                + "einer einzelnen Abfrage einer Vokabel angezeigt."));

        playMP3WhenTestingVocablesCheckBox = new JCheckBox(breakLongText(
                "Beim Abfragen Vokabeln automatisch vorlesen."));
        colorUserInputAtKanjiTestEvaluationDialogCheckBox = new JCheckBox(breakLongText(""
                + "Beim Abfragen der Kanji werden die Felder der Benutzereingaben im Dialog, "
                + "welcher Erfolg oder Misserfolg der Angaben zu einem einzelnen Kanji anzeigt, "
                + "rot oder grün eingefärbt."));


        showOnlyOwnListDialogOptions = false;
        applied = false;
    }

    private String breakLongText(String text) {
        return HtmlTool.htmlify(text, 100);
    }

    private ChangeButtonWithSizeChooser createNumberOfVocabularyBarColumnsChooser() {
        int rows = 2;
        int columns = 3;
        List<Integer> possibleSizes = CollectionsHelper.buildListFrom(
                1, 2, 3,
                4, 5, 6
                );

        Point parentLocation = new Point(getLocation());
        parentLocation.x += 300;
        parentLocation.y += 200;
        ChangeButtonWithSizeChooser chooser = new ChangeButtonWithSizeChooser(parentLocation,
                "Anzahl der Spalten mit den Vokabularien", rows, columns,
                possibleSizes);
        chooser.setSize(options.getNumberOfVocabularyBarColumns());
        chooser.differentSizesAreOk();

        return chooser;
    }

    private ChangeButtonWithSizeChooser createNumberOfColumnsOfBlattDarstellungChooser() {
        int rows = 2;
        int columns = 2;
        List<Integer> possibleSizes = CollectionsHelper.buildListFrom(
                1, 2,
                3, 4
                );

        Point parentLocation = new Point(getLocation());
        parentLocation.x += 300;
        parentLocation.y += 200;
        ChangeButtonWithSizeChooser chooser = new ChangeButtonWithSizeChooser(parentLocation,
                "Anzahl der Spalten in der Darstellung als Vokabelblatt", rows, columns,
                possibleSizes);
        chooser.setSize(options.getNumberOfVocabularyBarColumns());
        chooser.differentSizesAreOk();

        return chooser;
    }

    /**
     * Legt fest, dass nur die Optionen für den Dialog zur Bearbeitung eigener Listen angezeigt
     * werden.
     */
    public void showOnlyOwnListDialogOptions() {
        showOnlyOwnListDialogOptions = true;
    }

    /** Baut die Gui auf. */
    public void populate() {
        /*
         * Dies wird von außen aufgerufen, damit man vorher entscheiden kann, welche Bereiche des
         * Dialogs angezeigt werden sollen.
         */
        fillDialog();
    }

    /** Baut die Gui auf. */
    @Override
    protected void populateDialog() {
        initElements();

        if (showOnlyOwnListDialogOptions) {
            add(createOwnListEditorSelectionsPart(), BorderLayout.CENTER);
            add(createButtonsPartWithoutDefaultButton(),  BorderLayout.SOUTH);
        }
        else {
            add(sideListOrganiser.getPanel(), BorderLayout.CENTER);
            add(createButtonsPart(),  BorderLayout.SOUTH);
        }

        fillOptionValuesIntoGui(options);
        addActionListenersForEnableDisableOptions();
        enableOrDisableElements();
    }

    private void initElements() {
        initVocabularySortOrders();
        initVocablesShuffleTypes();
        initVocablesSubsetSelectionTypes();
        initEmptyTranslationsAcceptanceButtonGroup();
        initDaysButtons();
        initNumberOfRememberedOwnListsButton();
        initNumberOfShownTranslationsButton();
        initShowAllTranslationsCheckBox();
        initSideListOrganiser();
    }

    private void initVocabularySortOrders() {
        ButtonGroup group = new ButtonGroup();
        for (VocabularySortOrder sortOrder : VocabularySortOrder.values()) {
            vocabularySortOrders.add(sortOrder);
            JRadioButton button = new JRadioButton(sortOrder.getDescription());
            group.add(button);
            vocabularySortOrderRadioButtons.add(button);
        }
    }

    private void initVocablesShuffleTypes() {
        ButtonGroup group = new ButtonGroup();
        for (VocablesShuffleType vocablesShuffleType : VocablesShuffleType.values()) {
            vocablesShuffleTypes.add(vocablesShuffleType);
            JRadioButton button = new JRadioButton(vocablesShuffleType.getDescription());
            group.add(button);
            vocablesShuffleTypesRadioButtons.add(button);
        }
    }

    private void initVocablesSubsetSelectionTypes() {
        ButtonGroup group = new ButtonGroup();

        for (VocablesSubsetSelectionType vocablesSubsetSelectionType :
            VocablesSubsetSelectionType.values()) {
            vocablesSubsetSelectionTypes.add(vocablesSubsetSelectionType);
            JRadioButton button = new JRadioButton(vocablesSubsetSelectionType.getDescription());
            group.add(button);
            vocablesSubsetSelectionTypesRadioButtons.add(button);
        }
    }

    private void initEmptyTranslationsAcceptanceButtonGroup() {
        ButtonGroup group = new ButtonGroup();

        for (EmptyTranslationsAcceptance emptyTranslationsAcceptance :
            EmptyTranslationsAcceptance.values()) {
            emptyTranslationsAcceptances.add(emptyTranslationsAcceptance);
            JRadioButton button = new JRadioButton(
                    emptyTranslationsAcceptance.getOptionDescription());
            group.add(button);
            emptyTranslationsAcceptancesRadioButtons.add(button);
        }
    }

    private void initDaysButtons() {
        numberOfDaysForAWhileButton.addActionListener(e -> changeNumberOfDaysForAWhile());
        numberOfDaysAVocableIsNewButton.addActionListener(e -> changeNumberOfDaysAVocableIsNew());
    }

    private void changeNumberOfDaysForAWhile() {
        String description = "Anzahl Tage für 'lange nicht mehr' abgefragte Vokabeln";
        int rows = 3;
        int columns = 5;
        List<Integer> possibleSizes = CollectionsHelper.buildListFrom(
                14,  20,  25,  30,  31,
                40,  45,  50,  60,  70,
                75,  80,  90, 100, 120
                );
        int newSize = chooseSize(numberOfDaysForAWhile, description, rows, columns, possibleSizes);
        if (newSize != SizeChooser.NO_VALUE_SELECTED
                && newSize >= 0 && newSize <= 1000) {
            numberOfDaysForAWhile = newSize;
            setNumberOfDaysForAWhileButtonText();
        }
    }

    private void changeNumberOfDaysAVocableIsNew() {
        String description = "Anzahl Tage wie lange eine Vokabel als 'neu' gilt";
        int rows = 3;
        int columns = 5;
        List<Integer> possibleSizes = CollectionsHelper.buildListFrom(
                 1,   2,   5,   7,  10,
                14,  20,  25,  30,  31,
                40,  45,  50,  60,  70
                );
                int newSize = chooseSize(numberOfDaysAVocableIsNew, description, rows, columns,
                        possibleSizes);
                if (newSize != SizeChooser.NO_VALUE_SELECTED
                && newSize >= 0 && newSize <= 1000) {
            numberOfDaysAVocableIsNew = newSize;
            setNumberOfDaysAVocableIsNewButtonText();
        }
    }

    private int chooseSize(int actualSize, String description, int rows, int columns,
            List<Integer> possibleSizes) {
        SizeChooser chooser = new SizeChooser(actualSize, description, rows, columns, possibleSizes,
                getLocation(), null);
        chooser.setSize(actualSize);
        chooser.diffenrentSizesAreOk();
        chooser.setVisible(true);
        int newsize = chooser.getSize();
        return newsize;
    }

    private void initNumberOfRememberedOwnListsButton() {
        numberOfRememberedOwnListsButton.addActionListener(e -> changeNumberOfRememberedOwnLists());
    }

    private void changeNumberOfRememberedOwnLists() {
        String description = "Anzahl der im Listen-Commander angezeigten zuletzt besuchten Eigenen "
                + "Listen";
        int rows = 3;
        int columns = 7;
        List<Integer> possibleSizes = CollectionsHelper.buildListFrom(
                 3,   4,   5,   6,   7,   8,   9,
                10,  11,  12,  13,  14,  15,  16,
                17,  18,  19,  20,  25,  30,  35
                );
        int newSize = chooseSize(numberOfRememberedOwnLists, description, rows, columns,
                possibleSizes);
        if (newSize != SizeChooser.NO_VALUE_SELECTED
                && newSize >= 1 && newSize <= 100) {
            numberOfRememberedOwnLists = newSize;
            setNumberOfRememberedOwnListsButtonText();
        }
    }

    private void initNumberOfShownTranslationsButton() {
        numberOfShownTranslationsButton.addActionListener(e -> changeNumberOfShownTranslations());
    }

    private void changeNumberOfShownTranslations() {
        String description = "Anzahl angezeigter Übersetzungen in der Einzeldarstellung und bei "
                + "der Abfrage von Vokabeln";
        int rows = 3;
        int columns = 7;
        List<Integer> possibleSizes = CollectionsHelper.buildListFrom(
                 3,   4,   5,   6,   7,   8,   9,
                10,  11,  12,  13,  14,  15,  16,
                17,  18,  19,  20,  25,  30,  35
                );
        int newSize = chooseSize(numberOfShownTranslations, description, rows, columns,
                possibleSizes);
        if (newSize != SizeChooser.NO_VALUE_SELECTED
                && newSize >= 1 && newSize <= 100) {
            numberOfShownTranslations = newSize;
            setNumberOfShownTranslationsButtonText();
        }
    }

    private void initShowAllTranslationsCheckBox() {
        showAllTranslationsCheckBox.addActionListener(e -> showAllTranslationsChanged());
    }

    private void showAllTranslationsChanged() {
        numberOfShownTranslationsButton.setEnabled(!showAllTranslationsCheckBox.isSelected());
    }

    private void initSideListOrganiser() {
        sideListOrganiser.setListPanelWidth(300);

        sideListOrganiser.addElement("Pfade und Startoptionen", createPathesAndStartOptionsPanel());
        sideListOrganiser.addElement("Vokabeln anzeigen 1", createShowVocables1Panel());
        sideListOrganiser.addElement("Vokabeln anzeigen 2", createShowVocables2Panel());
        sideListOrganiser.addElement("Vokabeln abfragen 1", createTestVocable1Panel());
        sideListOrganiser.addElement("Vokabeln abfragen 2", createTestVocable2Panel());
        sideListOrganiser.addElement("Erfolg - Farben", createElevenColorsPanel());
        sideListOrganiser.addElement("Vokabularien", createVocabularyPanel());
        sideListOrganiser.addElement("Eigene Listen", createOwnListPanel());
        sideListOrganiser.addElement("Kanji abfragen", createTestKanjiPanel());

        sideListOrganiser.showFirstAddedContent();
    }

    private Component createPathesAndStartOptionsPanel() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(3, VerticalLayout.BOTH));

        panel.add(createPathesSelectionPart());
        panel.add(createCheckNewVersionsAtStartPart());
        panel.add(createOtherThingsAtStartPart());

        return panel;
    }

    private Component createTestVocable1Panel() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(3, VerticalLayout.BOTH));

        panel.add(createTestingOptionsSelectionPart());
        panel.add(createTestingJapaneseOptionsSelectionPart());
        panel.add(createTestingGermanOptionsSelectionPart());

        return panel;
    }

    private Component createTestVocable2Panel() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(3, VerticalLayout.BOTH));

        panel.add(createVocablesShuffleTypesPart());
        panel.add(createVocablesSubsetSelectionTypesPart());
        panel.add(createEmptyTranslationsAcceptancesPart());
        panel.add(numberOfDaysForAWhileButton);
        panel.add(numberOfDaysAVocableIsNewButton);

        return panel;
    }

    private Component createShowVocables1Panel() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(3, VerticalLayout.BOTH));

        panel.add(createViewingOptionsSelectionPart());
        panel.add(createTranslationsShownPart());
        panel.add(createBlattdarstellungSelectionsPart());

        return panel;
    }

    private Component createShowVocables2Panel() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(3, VerticalLayout.BOTH));

        panel.add(createVocableColorsPart());

        return panel;
    }

    private Component createVocabularyPanel() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(3, VerticalLayout.BOTH));

        panel.add(createVocabularyPart());
        panel.add(createVocabularyBarSelectionsPart());

        return panel;
    }

    private Component createElevenColorsPanel() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(3, VerticalLayout.BOTH));

        panel.add(elevenColorsUi.getPanel());

        return panel;
    }

    private Component createOwnListPanel() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(3, VerticalLayout.BOTH));

        panel.add(createOtherOwnListSelectionsPart());
        panel.add(createOwnListEditorSelectionsPart());

        return panel;
    }

    private void fillOptionValuesIntoGui(Options options) {
        vocabulariesPathSelection.setText(Text.nicePath(options.getVocabulariesPath()));
        switchToNextVocableAfterEntringTranslationCheckBox.setSelected(
                options.isSwitchToNextVocableAfterEntringTranslation());
        showKanaWhenTestingVocableCheckBox.setSelected(
                options.isShowKanaWhenTestingVocable());
        showKanjiWhenTestingVocableCheckBox.setSelected(
                options.isShowKanjiWhenTestingVocable());
        hideKanjiWhenTestingVocableAndKanaContainsOnlyHiraganaCheckBox.setSelected(
                options.isHideKanjiWhenTestingVocableAndKanaContainsOnlyHiragana());
        showRomajiWhenTestingVocableCheckBox.setSelected(
                options.isShowRomajiWhenTestingVocable());
        hideRomajiWhenTestingVocableAndKanaContainsOnlyHiraganaCheckBox.setSelected(
                options.isHideRomajiWhenTestingVocableAndKanaContainsOnlyHiragana());
        showPronunciationWhenTestingVocableCheckBox.setSelected(
                options.isShowPronunciationWhenTestingVocable());
        hidePronunciationWhenTestingVocableAndKanaContainsOnlyHiraganaCheckBox.setSelected(
                options.isHidePronunciationWhenTestingVocableAndKanaContainsOnlyHiragana());
        showCommentWhenTestingVocableCheckBox.setSelected(
                options.isShowCommentWhenTestingVocable());
        showVocabularyDescriptionWhenTestingVocableCheckBox.setSelected(
                options.isShowVocabularyDescriptionWhenTestingVocable());
        colorVocableDependingOnLastSuccessWhenTestingVocableCheckBox.setSelected(
                options.isColorVocableDependingOnLastSuccessWhenTestingVocable());
        colorVocableDependingOnLastSuccessWhenViewingVocableCheckBox.setSelected(
                options.isColorVocableDependingOnLastSuccessWhenViewingVocable());
        colorVocabularyDependingOnLastSuccessCheckBox.setSelected(
                options.isColorVocabularyDependingOnLastSuccess());
        hideStartOfVocabularyDescriptionCheckBox.setSelected(
                options.isHideStartOfVocabularyDescription());
        useTestOrderForViewingVocablesCheckBox.setSelected(
                options.isUseTestOrderForViewingVocables());
        allowMissingMp3CheckBox.setSelected(options.isAllowMissingMp3());
        reportMissingMp3CheckBox.setSelected(options.isReportMissingMp3());
        informAboutDoubleMp3AtStartupCheckBox.setSelected(options.isInformAboutDoubleMp3AtStartup());
        informAboutEqualKanaAndKanjiAtStartupCheckBox.setSelected(
                options.isInformAboutEqualKanaAndKanjiAtStartup());
        useOnlyFirstGermanTermTestingCheckBox.setSelected(
                options.isUseOnlyFirstGermanTermWhenTestingVocableGermanToJapanese());

        VocabularySortOrder sortOrder = options.getVocabularySortOrder();
        int index1 = vocabularySortOrders.indexOf(sortOrder);
        JRadioButton button1 = vocabularySortOrderRadioButtons.get(index1);
        button1.setSelected(true);

        VocablesShuffleType shuffleType = options.getVocablesShuffleType();
        int index2 = vocablesShuffleTypes.indexOf(shuffleType);
        JRadioButton button2 = vocablesShuffleTypesRadioButtons.get(index2);
        button2.setSelected(true);
        reverseShuffledVocablesCheckBox.setSelected(options.isReverseShuffledVocables());
        useVocablesShuffleTypeForShowingListsAndSheetsTooCheckBox.setSelected(
                options.isUseVocablesShuffleTypeForShowingListsAndSheetsToo());

        VocablesSubsetSelectionType subsetSelectionType = options.getVocablesSubsetSelectionType();
        int index3 = vocablesSubsetSelectionTypes.indexOf(subsetSelectionType);
        JRadioButton button3 = vocablesSubsetSelectionTypesRadioButtons.get(index3);
        button3.setSelected(true);

        showSuccessPercentInVocabularyBarCheckBox.setSelected(
                options.isShowSuccessPercentInVocabularyBar());

        numberOfVocabularyBarColumnsChooser.setSize(options.getNumberOfVocabularyBarColumns());
        numberOfColumnsOfBlattDarstellungChooser.setSize(
                options.getNumberOfColumnsOfBlattDarstellung());

        hideVocablesFromListInSearchAreaInOwnListEditorCheckBox.setSelected(
                options.isHideVocablesFromListInSearchAreaInOwnListEditor());

        showDetailsButtonInEinzeldarstellungCheckBox.setSelected(
                options.isShowDetailsButtonInEinzeldarstellung());
        showDetailsButtonInVokabelblattDarstellungCheckBox.setSelected(
                options.isShowDetailsButtonInVokabelblattDarstellung());
        hideLessStartOfVocabularyDescriptionForVhsCheckBox.setSelected(
                options.isHideLessStartOfVocabularyDescriptionForVhs());

        checkNewOwnListsVersionAtStartCheckBox.setSelected(
                options.isCheckNewOwnListsVersionAtStart());

        showKanaInKanjiSizeWhenTestingVocableCheckBox.setSelected(
                options.isShowKanaInKanjiSizeWhenTestingVocable());

        EmptyTranslationsAcceptance emptyTranslationsAcceptance =
                options.getEmptyTranslationsAcceptance();
        int index4 = emptyTranslationsAcceptances.indexOf(emptyTranslationsAcceptance);
        JRadioButton button4 = emptyTranslationsAcceptancesRadioButtons.get(index4);
        button4.setSelected(true);

        numberOfDaysForAWhile = options.getNumberOfDaysForAWhile();
        setNumberOfDaysForAWhileButtonText();

        numberOfDaysAVocableIsNew = options.getNumberOfDaysAVocableIsNew();
        setNumberOfDaysAVocableIsNewButtonText();

        elevenColorsUi.initWithOptions(options);
        vocablesColorsUi.initWithOptions(options);

        useSplitPaneInOwnListEditorCheckBox.setSelected(options.isUseSplitPaneInOwnListEditor());
        showTimestampsInStartupCheckBox.setSelected(options.isShowTimestampsInStartup());
        createGermanJapaneseTranslationAtStartupCheckBox.setSelected(
                options.isCreateGermanJapaneseTranslationAtStartup());

        informAboutEOrUInVocableFieldAusspracheAtStartupCheckBox.setSelected(
                options.isInformAboutEOrUInVocableFieldAusspracheAtStartup());

        numberOfRememberedOwnLists = options.getNumberOfRememberedOwnLists();
        setNumberOfRememberedOwnListsButtonText();

        showAllTranslationsCheckBox.setSelected(options.isShowAllTranslations());
        numberOfShownTranslations = options.getNumberOfShownTranslations();
        setNumberOfShownTranslationsButtonText();

        showNumberOfListsInListerCheckBox.setSelected(options.isShowNumberOfListsInLister());

        showDetailBottomButonsInEinzeldarstellungTooCheckBox.setSelected(
                options.isShowDetailBottomButonsInEinzeldarstellungToo());
        showDetailBottomButonsInTranslationEvaluationTooCheckBox.setSelected(
                options.isShowDetailBottomButonsInTranslationEvaluationToo());
        playMP3WhenTestingVocablesCheckBox.setSelected(
                options.isPlayMP3WhenTestingVocables());
        colorUserInputAtKanjiTestEvaluationDialogCheckBox.setSelected(
                options.isColorUserInputAtKanjiTestEvaluationDialog());
    }

    private void setNumberOfDaysForAWhileButtonText() {
        numberOfDaysForAWhileButton.setText(
                "Anzahl Tage für 'lange nicht mehr' abgefragte Vokabeln, momentan "
                        + numberOfDaysForAWhile);
    }

    private void setNumberOfDaysAVocableIsNewButtonText() {
        numberOfDaysAVocableIsNewButton.setText(
                "Anzahl Tage wie lange eine Vokabel als 'neu' gilt, momentan "
                        + numberOfDaysAVocableIsNew);
    }

    private void setNumberOfRememberedOwnListsButtonText() {
        numberOfRememberedOwnListsButton.setText("Anzahl der gemerkten zuletzt besuchten eigenen "
                + "Listen, momentan " + numberOfRememberedOwnLists);
    }

    private void setNumberOfShownTranslationsButtonText() {
        numberOfShownTranslationsButton.setText("Anzahl der angezeigten Übersetzungen in der "
                + "Einzeldarstellung und bei der Abfrage, momentan " + numberOfShownTranslations);
    }

    private void addActionListenersForEnableDisableOptions() {
        showKanjiWhenTestingVocableCheckBox.addActionListener(e -> enableOrDisableKanji());
        showRomajiWhenTestingVocableCheckBox.addActionListener(e -> enableOrDisableRomaji());
        showPronunciationWhenTestingVocableCheckBox.addActionListener(
                e -> enableOrDisablePronunciation());
        hideStartOfVocabularyDescriptionCheckBox.addActionListener(
                e -> enableOrDisableHideStartOfVocabularyDescription());
    }

    private void enableOrDisableElements() {
        enableOrDisableKanji();
        enableOrDisableRomaji();
        enableOrDisablePronunciation();
        enableOrDisableHideStartOfVocabularyDescription();
    }

    private void enableOrDisableKanji() {
        boolean enabled = showKanjiWhenTestingVocableCheckBox.isSelected();
        hideKanjiWhenTestingVocableAndKanaContainsOnlyHiraganaCheckBox.setEnabled(enabled);
    }

    private void enableOrDisableRomaji() {
        boolean enabled = showRomajiWhenTestingVocableCheckBox.isSelected();
        hideRomajiWhenTestingVocableAndKanaContainsOnlyHiraganaCheckBox.setEnabled(enabled);
    }

    private void enableOrDisablePronunciation() {
        boolean enabled = showPronunciationWhenTestingVocableCheckBox.isSelected();
        hidePronunciationWhenTestingVocableAndKanaContainsOnlyHiraganaCheckBox.setEnabled(enabled);
    }

    private void enableOrDisableHideStartOfVocabularyDescription() {
        boolean enabled = hideStartOfVocabularyDescriptionCheckBox.isSelected();
        hideLessStartOfVocabularyDescriptionForVhsCheckBox.setEnabled(enabled);
    }

    private Component createPathesSelectionPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(3, VerticalLayout.BOTH));
        GuiTools.createTitle("Pfade", panel);

        panel.add(vocabulariesPathSelection.getPanel());

        return panel;
    }

    private Component createCheckNewVersionsAtStartPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(3, VerticalLayout.BOTH));
        GuiTools.createTitle("Prüfung auf Updates beim Start des Programms", panel);

        panel.add(checkNewOwnListsVersionAtStartCheckBox);

        return panel;
    }

    private Component createOtherThingsAtStartPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(5, VerticalLayout.BOTH)); // 3
        GuiTools.createTitle("Prüfungen etc. beim Start des Programms", panel);

        panel.add(informAboutDoubleMp3AtStartupCheckBox);
        panel.add(informAboutEqualKanaAndKanjiAtStartupCheckBox);
        panel.add(allowMissingMp3CheckBox);
        panel.add(reportMissingMp3CheckBox);
        panel.add(informAboutEOrUInVocableFieldAusspracheAtStartupCheckBox);
        panel.add(showTimestampsInStartupCheckBox);
        panel.add(createGermanJapaneseTranslationAtStartupCheckBox);

        return panel;
    }

    private Component createVocabularyPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(3, VerticalLayout.BOTH));

        panel.add(createVocabularyBaseOptionsPart());
        panel.add(createVocabularySortOrderPart());

        return panel;
    }

    private Component createVocabularyBaseOptionsPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(3, VerticalLayout.BOTH));
        GuiTools.createTitle("Grundeinstellungen der Vokabularien", panel);

        panel.add(colorVocabularyDependingOnLastSuccessCheckBox);
        panel.add(showSuccessPercentInVocabularyBarCheckBox);
        panel.add(hideStartOfVocabularyDescriptionCheckBox);
        panel.add(hideLessStartOfVocabularyDescriptionForVhsCheckBox);

        return panel;
    }

    private Component createVocabularySortOrderPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(3, VerticalLayout.BOTH));
        GuiTools.createTitle("Sortierung der Vokabularien", panel);

        for (JRadioButton button : vocabularySortOrderRadioButtons) {
            panel.add(button);
        }

        return panel;
    }

    private Component createTestingOptionsSelectionPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(3, VerticalLayout.BOTH));
        GuiTools.createTitle("Abfragen von Vokablen - generell", panel);

        panel.add(switchToNextVocableAfterEntringTranslationCheckBox);
        panel.add(colorVocableDependingOnLastSuccessWhenTestingVocableCheckBox);
        panel.add(showDetailBottomButonsInTranslationEvaluationTooCheckBox);

        return panel;
    }

    private Component createTestingJapaneseOptionsSelectionPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(3, VerticalLayout.BOTH));
        GuiTools.createTitle("Abfragen von japanischen Vokablen", panel);
        panel.add(showKanaWhenTestingVocableCheckBox);
        panel.add(showKanjiWhenTestingVocableCheckBox);
        panel.add(hideKanjiWhenTestingVocableAndKanaContainsOnlyHiraganaCheckBox);
        panel.add(showRomajiWhenTestingVocableCheckBox);
        panel.add(hideRomajiWhenTestingVocableAndKanaContainsOnlyHiraganaCheckBox);
        panel.add(showPronunciationWhenTestingVocableCheckBox);
        panel.add(hidePronunciationWhenTestingVocableAndKanaContainsOnlyHiraganaCheckBox);
        panel.add(showCommentWhenTestingVocableCheckBox);
        panel.add(showVocabularyDescriptionWhenTestingVocableCheckBox);
        panel.add(showKanaInKanjiSizeWhenTestingVocableCheckBox);
        panel.add(playMP3WhenTestingVocablesCheckBox);

        return panel;
    }

    private Component createTestingGermanOptionsSelectionPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(3, VerticalLayout.BOTH));
        GuiTools.createTitle("Abfragen von deutschen Vokablen", panel);

        panel.add(useOnlyFirstGermanTermTestingCheckBox);

        return panel;
    }

    private Component createVocablesShuffleTypesPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(3, VerticalLayout.BOTH));
        GuiTools.createTitle("Sortierung von Vokabellisten für die Abfrage", panel);

        panel.add(createVocablesShuffleTypesOptionsPart());
        panel.add(createVocablesShuffleTypesRadioButtonsPart());

        return panel;
    }

    private Component createVocablesShuffleTypesOptionsPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(3, VerticalLayout.BOTH));
        GuiTools.createTitle(panel);

        panel.add(reverseShuffledVocablesCheckBox);
        panel.add(useVocablesShuffleTypeForShowingListsAndSheetsTooCheckBox);

        return panel;
    }

    private Component createVocablesShuffleTypesRadioButtonsPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(3, VerticalLayout.BOTH));
        GuiTools.createTitle(panel);

        for (JRadioButton button : vocablesShuffleTypesRadioButtons) {
            panel.add(button);
        }

        return panel;
    }

    private Component createVocablesSubsetSelectionTypesPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(3, VerticalLayout.BOTH));
        GuiTools.createTitle("Auswahl von Teilmengen von Vokabellisten für die Abfrage", panel);

        for (JRadioButton button : vocablesSubsetSelectionTypesRadioButtons) {
            panel.add(button);
        }

        return panel;
    }

    private Component createEmptyTranslationsAcceptancesPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(3, VerticalLayout.BOTH));
        GuiTools.createTitle("Wann soll auf die Eingabe einer leeren Übersetzung reagiert werden?",
                panel);

        for (JRadioButton button : emptyTranslationsAcceptancesRadioButtons) {
            panel.add(button);
        }

        return panel;
    }

    private Component createViewingOptionsSelectionPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(3, VerticalLayout.BOTH));
        GuiTools.createTitle("Anzeigen von Vokablen", panel);

        panel.add(useTestOrderForViewingVocablesCheckBox);
        panel.add(colorVocableDependingOnLastSuccessWhenViewingVocableCheckBox);
        panel.add(showNumberOfListsInListerCheckBox);
        panel.add(showDetailBottomButonsInEinzeldarstellungTooCheckBox);
        panel.add(showDetailsButtonInEinzeldarstellungCheckBox);
        panel.add(showDetailsButtonInVokabelblattDarstellungCheckBox);

        return panel;
    }

    private Component createVocableColorsPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());
        GuiTools.createTitle("Farben bei der Darstellung einer Vokabel", panel);

        panel.add(vocablesColorsUi.getPanel(), BorderLayout.CENTER);

        return panel;
    }

    private Component createTranslationsShownPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(3, VerticalLayout.BOTH));
        GuiTools.createTitle("Anzeige der Übersetzungen in der Einzeldarstellung und bei der "
                + "Bewertung der Abfrage von Vokabeln", panel);

        panel.add(showAllTranslationsCheckBox);
        panel.add(numberOfShownTranslationsButton);

        return panel;
    }

    private Component createBlattdarstellungSelectionsPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(3, VerticalLayout.BOTH));
        GuiTools.createTitle("Blattdarstellung", panel);

        panel.add(numberOfColumnsOfBlattDarstellungChooser.getSizeChangeButton());

        return panel;
    }

    private Component createVocabularyBarSelectionsPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(3, VerticalLayout.BOTH));
        GuiTools.createTitle("Sonstige Optionen für Vokabularien", panel);

        panel.add(numberOfVocabularyBarColumnsChooser.getSizeChangeButton());

        return panel;
    }

    private Component createOtherOwnListSelectionsPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(3, VerticalLayout.BOTH));
        GuiTools.createTitle("Sonstige Optionen für eigene Listen", panel);

        panel.add(numberOfRememberedOwnListsButton);

        return panel;
    }

    private Component createOwnListEditorSelectionsPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(3, VerticalLayout.BOTH));
        GuiTools.createTitle("Optionen für den Editor zur Bearbeitung eigener Listen", panel);

        panel.add(hideVocablesFromListInSearchAreaInOwnListEditorCheckBox);
        panel.add(useSplitPaneInOwnListEditorCheckBox);

        return panel;
    }

    private Component createTestKanjiPanel() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(3, VerticalLayout.BOTH));

        panel.add(createColorKanjiTestPart());

        return panel;
    }

    private Component createColorKanjiTestPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(3, VerticalLayout.BOTH));
        GuiTools.createTitle("Optionen für das Einfärben beim Testen von Kanji", panel);

        panel.add(colorUserInputAtKanjiTestEvaluationDialogCheckBox);

        return panel;
    }

    private Component createButtonsPartWithoutDefaultButton() {
        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());

        panel.add(createQuitButton(), BorderLayout.WEST);
        panel.add(createOkButton(), BorderLayout.EAST);

        return panel;
    }

    private Component createButtonsPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());

        panel.add(createLeftButtonsPart(), BorderLayout.WEST);
        panel.add(createOkButton(), BorderLayout.EAST);

        return panel;
    }

    private Component createLeftButtonsPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());

        panel.add(createQuitButton(), BorderLayout.WEST);
        panel.add(createToDefaultButton(), BorderLayout.EAST);

        return panel;
    }

    private Component createQuitButton() {
        JButton button = new JButton("Abbrechen");
        button.addActionListener(e -> closeDialog());
        return button;
    }

    private Component createToDefaultButton() {
        JButton button = new JButton("Defaultwerte");
        button.addActionListener(e -> toDefault());
        return button;
    }

    private void toDefault() {
        if (GuiTools.askUser(getDialog(), "Werte auf Default setzen?",
                "Sollen alle Optionen auf den Default-Wert gesetzt werden?")) {

            String vocabulariesPath = options.getVocabulariesPath();
            Options newOptions = new Options(new VocabularyTrainerVersion().getVersion());
            newOptions.setVocabulariesPath(vocabulariesPath);

            fillOptionValuesIntoGui(newOptions);
        }
    }

    private Component createOkButton() {
        JButton button = new JButton("Änderungen übernehmen");
        button.addActionListener(e -> apply());
        return button;
    }

    /**
     * Wertet nach dem Klick auf 'Änderungen übernehmen' die gewählten Einstellungen aus und
     * schreibt diese in die Optionen.
     */
    private void apply() {
        String vocabulariesPath = vocabulariesPathSelection.getText();
        boolean switchToNextVocableAfterEntringTranslation =
                switchToNextVocableAfterEntringTranslationCheckBox.isSelected();
        boolean showKanaWhenTestingVocable = showKanaWhenTestingVocableCheckBox.isSelected();
        boolean showKanjiWhenTestingVocable = showKanjiWhenTestingVocableCheckBox.isSelected();
        boolean hideKanjiWhenTestingVocableAndKanaContainsOnlyHiragana =
                hideKanjiWhenTestingVocableAndKanaContainsOnlyHiraganaCheckBox.isSelected();
        boolean showRomajiWhenTestingVocable =
                showRomajiWhenTestingVocableCheckBox.isSelected();
        boolean hideRomajiWhenTestingVocableAndKanaContainsOnlyHiragana =
                hideRomajiWhenTestingVocableAndKanaContainsOnlyHiraganaCheckBox.isSelected();
        boolean showPronunciationWhenTestingVocable =
                showPronunciationWhenTestingVocableCheckBox.isSelected();
        boolean hidePronunciationWhenTestingVocableAndKanaContainsOnlyHiragana =
                hidePronunciationWhenTestingVocableAndKanaContainsOnlyHiraganaCheckBox.isSelected();
        boolean showCommentWhenTestingVocable =
                showCommentWhenTestingVocableCheckBox.isSelected();
        boolean showVocabularyDescriptionWhenTestingVocable =
                showVocabularyDescriptionWhenTestingVocableCheckBox.isSelected();
        boolean colorVocableDependingOnLastSuccessWhenTestingVocable =
                colorVocableDependingOnLastSuccessWhenTestingVocableCheckBox.isSelected();
        boolean colorVocableDependingOnLastSuccessWhenViewingVocable =
                colorVocableDependingOnLastSuccessWhenViewingVocableCheckBox.isSelected();
        boolean colorVocabularyDependingOnLastSuccess =
                colorVocabularyDependingOnLastSuccessCheckBox.isSelected();
        boolean hideStartOfVocabularyDescription =
                hideStartOfVocabularyDescriptionCheckBox.isSelected();
        boolean useTestOrderForViewingVocables =
                useTestOrderForViewingVocablesCheckBox.isSelected();
        boolean hideVocablesFromListInSearchAreaInOwnListEditor =
                hideVocablesFromListInSearchAreaInOwnListEditorCheckBox.isSelected();
        boolean allowMissingMp3 = allowMissingMp3CheckBox.isSelected();
        boolean reportMissingMp3 = reportMissingMp3CheckBox.isSelected();
        boolean informAboutDoubleMp3AtStartup = informAboutDoubleMp3AtStartupCheckBox.isSelected();
        boolean informAboutEqualKanaAndKanjiAtStartup =
                informAboutEqualKanaAndKanjiAtStartupCheckBox.isSelected();
        boolean useOnlyFirstGermanTermWhenTestingVocableGermanToJapanese =
                useOnlyFirstGermanTermTestingCheckBox.isSelected();
        boolean showDetailsButtonInEinzeldarstellung =
                showDetailsButtonInEinzeldarstellungCheckBox.isSelected();
        boolean showDetailsButtonInVokabelblattDarstellung =
                showDetailsButtonInVokabelblattDarstellungCheckBox.isSelected();
        boolean hideLessStartOfVocabularyDescriptionForVhs =
                hideLessStartOfVocabularyDescriptionForVhsCheckBox.isSelected();
        boolean checkNewOwnListsVersionAtStart =
                checkNewOwnListsVersionAtStartCheckBox.isSelected();
        boolean showAllTranslations = showAllTranslationsCheckBox.isSelected();
        boolean showNumberOfListsInLister = showNumberOfListsInListerCheckBox.isSelected();
        boolean showDetailBottomButonsInEinzeldarstellungToo =
                showDetailBottomButonsInEinzeldarstellungTooCheckBox.isSelected();
        boolean showDetailBottomButonsInTranslationEvaluationToo =
                showDetailBottomButonsInTranslationEvaluationTooCheckBox.isSelected();
        boolean playMP3WhenTestingVocables = playMP3WhenTestingVocablesCheckBox.isSelected();
        boolean colorUserInputAtKanjiTestEvaluationDialog =
                colorUserInputAtKanjiTestEvaluationDialogCheckBox.isSelected();


        VocabularySortOrder sortOrder = options.getVocabularySortOrder(); // nur Initialisierung
        for (int index = 0; index < vocabularySortOrderRadioButtons.size(); ++index) {
            JRadioButton button = vocabularySortOrderRadioButtons.get(index);
            if (button.isSelected()) {
                sortOrder = vocabularySortOrders.get(index);
                break;
            }
        }

        VocablesShuffleType shuffleType = options.getVocablesShuffleType(); // nur Initialisierung
        for (int index = 0; index < vocablesShuffleTypesRadioButtons.size(); ++index) {
            JRadioButton button = vocablesShuffleTypesRadioButtons.get(index);
            if (button.isSelected()) {
                shuffleType = vocablesShuffleTypes.get(index);
                break;
            }
        }
        boolean reverseShuffledVocables = reverseShuffledVocablesCheckBox.isSelected();
        boolean useVocablesShuffleTypeForShowingListsAndSheetsToo =
                useVocablesShuffleTypeForShowingListsAndSheetsTooCheckBox.isSelected();

        VocablesSubsetSelectionType subsetSelectionType =
                options.getVocablesSubsetSelectionType(); // nur Initialisierung
        for (int index = 0; index < vocablesSubsetSelectionTypesRadioButtons.size(); ++index) {
            JRadioButton button = vocablesSubsetSelectionTypesRadioButtons.get(index);
            if (button.isSelected()) {
                subsetSelectionType = vocablesSubsetSelectionTypes.get(index);
                break;
            }
        }

        boolean showSuccessPercentInVocabularyBar =
                showSuccessPercentInVocabularyBarCheckBox.isSelected();
        int numberOfVocabularyBarColumns = numberOfVocabularyBarColumnsChooser.getSize();
        int numberOfColumnsOfBlattDarstellung = numberOfColumnsOfBlattDarstellungChooser.getSize();

        boolean showKanaInKanjiSizeWhenTestingVocable =
                showKanaInKanjiSizeWhenTestingVocableCheckBox.isSelected();

        EmptyTranslationsAcceptance emptyTranslationsAcceptance =
                options.getEmptyTranslationsAcceptance(); // nur Initialisierung
        for (int index = 0; index < emptyTranslationsAcceptancesRadioButtons.size(); ++index) {
            JRadioButton button = emptyTranslationsAcceptancesRadioButtons.get(index);
            if (button.isSelected()) {
                emptyTranslationsAcceptance = emptyTranslationsAcceptances.get(index);
                break;
            }
        }

        boolean useSplitPaneInOwnListEditor = useSplitPaneInOwnListEditorCheckBox.isSelected();
        boolean showTimestampsInStartup = showTimestampsInStartupCheckBox.isSelected();
        boolean createGermanJapaneseTranslationAtStartup =
                createGermanJapaneseTranslationAtStartupCheckBox.isSelected();

        boolean informAboutEOrUInVocableFieldAusspracheAtStartup =
                informAboutEOrUInVocableFieldAusspracheAtStartupCheckBox.isSelected();


        options.setVocabulariesPath(Text.nicePath(vocabulariesPath));
        options.setSwitchToNextVocableAfterEntringTranslation(
                switchToNextVocableAfterEntringTranslation);
        options.setShowKanaWhenTestingVocable(showKanaWhenTestingVocable);
        options.setShowKanjiWhenTestingVocable(showKanjiWhenTestingVocable);
        options.setHideKanjiWhenTestingVocableAndKanaContainsOnlyHiragana(
                hideKanjiWhenTestingVocableAndKanaContainsOnlyHiragana);
        options.setShowRomajiWhenTestingVocable(showRomajiWhenTestingVocable);
        options.setHideRomajiWhenTestingVocableAndKanaContainsOnlyHiragana(
                hideRomajiWhenTestingVocableAndKanaContainsOnlyHiragana);
        options.setShowPronunciationWhenTestingVocable(showPronunciationWhenTestingVocable);
        options.setHidePronunciationWhenTestingVocableAndKanaContainsOnlyHiragana(
                hidePronunciationWhenTestingVocableAndKanaContainsOnlyHiragana);
        options.setShowCommentWhenTestingVocable(showCommentWhenTestingVocable);
        options.setShowVocabularyDescriptionWhenTestingVocable(
                showVocabularyDescriptionWhenTestingVocable);
        options.setColorVocableDependingOnLastSuccessWhenTestingVocable(
                colorVocableDependingOnLastSuccessWhenTestingVocable);
        options.setColorVocableDependingOnLastSuccessWhenViewingVocable(
                colorVocableDependingOnLastSuccessWhenViewingVocable);
        options.setVocabularySortOrder(sortOrder);
        options.setColorVocabularyDependingOnLastSuccess(colorVocabularyDependingOnLastSuccess);
        options.setVocablesShuffleType(shuffleType);
        options.setVocablesSubsetSelectionType(subsetSelectionType);
        options.setReverseShuffledVocables(reverseShuffledVocables);
        options.setUseVocablesShuffleTypeForShowingListsAndSheetsToo(
                useVocablesShuffleTypeForShowingListsAndSheetsToo);
        options.setHideStartOfVocabularyDescription(hideStartOfVocabularyDescription);
        options.setUseTestOrderForViewingVocables(useTestOrderForViewingVocables);
        options.setShowSuccessPercentInVocabularyBar(showSuccessPercentInVocabularyBar);
        options.setNumberOfVocabularyBarColumns(numberOfVocabularyBarColumns);
        options.setHideVocablesFromListInSearchAreaInOwnListEditor(
                hideVocablesFromListInSearchAreaInOwnListEditor);
        options.setAllowMissingMp3(allowMissingMp3);
        options.setReportMissingMp3(reportMissingMp3);
        options.setNumberOfColumnsOfBlattDarstellung(numberOfColumnsOfBlattDarstellung);
        options.setInformAboutDoubleMp3AtStartup(informAboutDoubleMp3AtStartup);
        options.setInformAboutEqualKanaAndKanjiAtStartup(informAboutEqualKanaAndKanjiAtStartup);
        options.setUseOnlyFirstGermanTermWhenTestingVocableGermanToJapanese(
                useOnlyFirstGermanTermWhenTestingVocableGermanToJapanese);
        options.setShowDetailsButtonInEinzeldarstellung(showDetailsButtonInEinzeldarstellung);
        options.setShowDetailsButtonInVokabelblattDarstellung(
                showDetailsButtonInVokabelblattDarstellung);
        options.setHideLessStartOfVocabularyDescriptionForVhs(
                hideLessStartOfVocabularyDescriptionForVhs);
        options.setCheckNewOwnListsVersionAtStart(checkNewOwnListsVersionAtStart);
        options.setShowKanaInKanjiSizeWhenTestingVocable(showKanaInKanjiSizeWhenTestingVocable);
        options.setEmptyTranslationsAcceptance(emptyTranslationsAcceptance);
        options.setNumberOfDaysForAWhile(numberOfDaysForAWhile);
        options.setNumberOfDaysAVocableIsNew(numberOfDaysAVocableIsNew);
        elevenColorsUi.setIntoOptions(options);
        options.setUseSplitPaneInOwnListEditor(useSplitPaneInOwnListEditor);
        vocablesColorsUi.setIntoOptions(options);
        options.setShowTimestampsInStartup(showTimestampsInStartup);
        options.setCreateGermanJapaneseTranslationAtStartup(
                createGermanJapaneseTranslationAtStartup);
        options.setInformAboutEOrUInVocableFieldAusspracheAtStartup(
                informAboutEOrUInVocableFieldAusspracheAtStartup);
        options.setShowAllTranslations(showAllTranslations);
        options.setNumberOfRememberedOwnLists(numberOfRememberedOwnLists);
        options.setNumberOfShownTranslations(numberOfShownTranslations);
        options.setShowNumberOfListsInLister(showNumberOfListsInLister);
        options.setShowDetailBottomButonsInEinzeldarstellungToo(
                showDetailBottomButonsInEinzeldarstellungToo);
        options.setShowDetailBottomButonsInTranslationEvaluationToo(
                showDetailBottomButonsInTranslationEvaluationToo);
        options.setPlayMP3WhenTestingVocables(playMP3WhenTestingVocables);
        options.setColorUserInputAtKanjiTestEvaluationDialog(
                colorUserInputAtKanjiTestEvaluationDialog);

        applied = true;
        closeDialog();
    }

    /** Gibt an, ob der Dialog mit Klick auf OK beendet wurde. */
    public boolean wasApplied() {
        return applied;
    }

}
