package de.duehl.basics.io.textfile;

import de.duehl.basics.caller.CallerDeterminer;

/*
 * Copyright 2020 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import de.duehl.basics.io.Charset;

/**
 * Diese Klasse liest eine Liste mit den Texten aus den Zeilen einer Textdatei im Jar ein.
 *
 * Die Zeilen werden getrimmt. Dabei werden leere Zeilen und solche, die mit dem Kommentarzeichen
 * (ggf., nach Leerraum) überlesen. Kommt in einer Datenzeile weiter hinten das Kommentarzeichen
 * vor, so wird ab dort abgeschnitten und getrimmt.
 *
 * @version 1.01     2020-01-28
 * @author Christian Dühl
 */

public class StringsFromTextFileInJarReader extends StringsFromTextFileReader {

    /** Kommentarzeichen '#' für Konstruktoren ohne diese Angabe. */
    private static final String DEFAULT_COMMENT_CHARACTER = "#";

    /**
     * Konstruktor für UTF-8 Files mit Standard-Kommentarzeichen '#' und der Klasse des Aufrufers,
     * neben der also die Textdatei erwartet wird.
     *
     * @param filename
     *            Name der einzulesenden Datei ohne Pfad.
     */
    public StringsFromTextFileInJarReader(String filename) {
        this(filename, Charset.UTF_8, CallerDeterminer.determineCallingClass(),
                DEFAULT_COMMENT_CHARACTER);
    }

    /**
     * Konstruktor für UTF-8 Files mit Standard-Kommentarzeichen '#'.
     *
     * @param filename
     *            Name der einzulesenden Datei ohne Pfad.
     * @param clazz
     *            Klasse neben der die Datei sich im befindet.
     */
    public StringsFromTextFileInJarReader(String filename, Class<?> clazz) {
        this(filename, Charset.UTF_8, clazz, DEFAULT_COMMENT_CHARACTER);
    }

    /**
     * Konstruktor.
     *
     * @param filename
     *            Name der einzulesenden Datei ohne Pfad.
     * @param charset
     *            Kodierung der einzulesenden Datei.
     * @param clazz
     *            Klasse neben der die Datei sich im befindet.
     * @param commentChar
     *            Das Kommentarzeichen.
     */
    public StringsFromTextFileInJarReader(String filename, Charset charset, Class<?> clazz,
            String commentChar) {
        super(new FullTextFileFromJarReader(filename, charset, clazz), commentChar);
    }

}
