package de.duehl.basics.io.textfile.split;

/*
 * Copyright 2022 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.io.File;
import java.io.FilenameFilter;
import java.util.List;

import de.duehl.basics.collections.CollectionsHelper;
import de.duehl.basics.io.Charset;
import de.duehl.basics.io.FileHelper;
import de.duehl.basics.text.Text;

/**
 * Diese Klasse teilt eine Datei auf und setzt sie wieder zusammen.
 *
 * @version 1.01     2022-02-22
 * @author Christian Dühl
 */

public class SplitAndJoinHelper {

    /**
     * Teilt eine Datei mit Titeln in 500er-Blöcke auf.
     *
     * @param filename
     *            Name der aufzuteilenden Datei.
     */
    public static void splitWithTitlesTo500erParts(String filename) {
        Charset charset = Charset.ISO_8859_1;

        int size = 500;
        boolean hasTitles = true;

        TextFileSplitter splitter = new TextFileSplitter(filename, charset, size, hasTitles);
        splitter.work();

        List<String> list = splitter.getFilenamesOfCreatedSplitFiles();
        CollectionsHelper.printElements(list);
    }

    /**
     * Setzt eine aufgeteilte Datei wieder zusammen.
     *
     * Gesucht werden zur Eingabe "foo.txt" alle Dateien der Art
     * "foo_joined_IRGENDWAS_DDDD_IRGENDWAS.txt" in denen kein "_backup" vorkommt. Diese werden in
     * der passenden Reihenfolge zusammengesetzt und in einer Datei mit dem Namen "foo_joined.txt"
     * abgelegt.
     *
     * @param filename
     *            Dateiname ohne numerische Angaben vor der Erweiterung.
     */
    public static void join(String filename) {
        String dir = FileHelper.getDirName(filename);
        String joinedFilename = FileHelper.insertBeforeExtension(filename, "_joined");
        String filenameStart = FileHelper.getBareName(
                Text.removeTextAtEndIfEndsWith(filename, ".txt"));

        List<String> filenamesOfSplitFiles = FileHelper.findInDir(dir,
                new FilenameFilter() {
                    @Override
                    public boolean accept(File dir, String name) {
                        return name.startsWith(filenameStart)
                                && name.matches(".*_\\d{4}.*")
                                && name.endsWith(".txt")
                                && !name.contains("_backup");
                    }
                });

        Charset charset = Charset.ISO_8859_1;
        boolean hasTitles = true;
        SplittedTextFileJoiner joiner = new SplittedTextFileJoiner(filenamesOfSplitFiles,
                joinedFilename, charset, hasTitles);
        //joiner.beQuiet();
        joiner.work();

        Text.say("Zusammengefügt aus " + filenamesOfSplitFiles.size() + " Dateien.");
        Text.say("Geschrieben nach " + joinedFilename + ".");
    }

}
