package de.duehl.basics.logging.eraser;

/*
 * Copyright 2020 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.io.File;
import java.io.FilenameFilter;

import de.duehl.basics.datetime.time.TimeHelper;
import de.duehl.basics.logging.Logger;

/**
 * Dieses Klasse entfernt Logdateien, die älter als sieben Tage sind.
 *
 * @version 1.01     2020-03-11
 * @author Christian Dühl
 */

public class LogEraser {

    /** Pfad zu den Logdateien. */
    private final String path;

    /** Anfang der Namen der zu löschenden Logdateien. */
    private final String begin;

    /** Ende der Namen der zu löschenden Logdateien. */
    private final String end;

    /** Logger zum Ausgeben der Ereignisse. */
    private final Logger logger;

    /** Anzahl der Tage, wie lange ein Logfile ungelöscht bestehen bleiben darf. */
    private int days;

    /**
     * Konstruktor, Ausgaben werden nach System.out geschrieben.
     *
     * @param path
     *            Pfad zu den Logdateien.
     * @param begin
     *            Anfang der Namen der zu löschenden Logdateien.
     * @param end
     *            Ende der Namen der zu löschenden Logdateien.
     */
    public LogEraser(String path, String begin, String end) {
        this(path, begin, end, new StdOutLogger());
    }

    /**
     * Konstruktor, Ausgaben werden in den übergebenen Logger geschrieben.
     *
     * @param path
     *            Pfad zu den Logdateien.
     * @param begin
     *            Anfang der Namen der zu löschenden Logdateien.
     * @param end
     *            Ende der Namen der zu löschenden Logdateien.
     * @param logger
     *            Logger zum Ausgeben der Ereignisse.
     */
    public LogEraser(String path, String begin, String end, Logger logger) {
        this.path = path;
        this.begin = begin;
        this.end = end;
        this.logger = logger;
        days = 7;
    }

    /** Gibt die Anzahl der Tage, wie lange ein Logfile ungelöscht bestehen bleiben darf, zurück. */
    public int getDays() {
        return days;
    }

    /** Setzt die Anzahl der Tage, wie lange ein Logfile ungelöscht bestehen bleiben darf. */
    public void setDays(int days) {
        this.days = days;
    }

    /** Entfernt die Logdateien. */
    public void erase() {
        logger.log("Start erasing old files in '" + path + "' with beginning '" + begin
                + "' and end '" + end + "'.");

        File dir = new File(path);
        long now = System.currentTimeMillis();
        long sevenDays = days * TimeHelper.ONE_DAY_MILLIS;
        int countDeleted = 0;
        for (File file : dir.listFiles(createLogFilenameFilter())) {
            long age = now - file.lastModified();
            if (age > sevenDays) {
                boolean success = file.delete();
                if (success) {
                    logger.log("Erasing file " + file);
                    ++countDeleted;
                }
                else {
                    logger.log("Can't erase file " + file + "!");
                }
            }
        }

        logger.log("End erasing old files in '" + path + "' with beginning '" + begin
                + "' and end '" + end + "'. Number of deleted files: " + countDeleted);
    }

    /** Erzeugt einen FilenameFilter passend zu Anfang und Ende der Namen der Logdateien. */
    private FilenameFilter createLogFilenameFilter() {
        return new FilenameFilter() {
            @Override
            public boolean accept(File dir, String name) {
                return new File(dir, name).isFile()
                        && name.startsWith(begin)
                        && name.endsWith(end);
            }
        };
    }

    /** Stringrepräsentation. */
    @Override
    public String toString() {
        return "LogEraser[path='" + path + "', begin='" + begin + "', end='" + end + "', logger='"
                + logger + "']";
    }

}
