package de.duehl.basics.system.launcher;

import java.util.List;

import de.duehl.basics.collections.CollectionsHelper;

/*
 * Copyright 2024 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import de.duehl.basics.io.FileHelper;
import de.duehl.basics.logic.PrintErrorHandler;
import de.duehl.basics.system.SystemTools;
import de.duehl.basics.system.starter.CmdStarter;
import de.duehl.basics.text.Text;

/**
 * Diese Klasse startet eine Jar mit Hilfe des gleichen Java-Executables, welches auch das laufende
 * Programm gestartet hat.
 *
 * Startet man javaw.exe, so sieht man danach deren Ausgaben auf der Kommandozeile nicht.
 *
 * @version 1.01     2024-12-08
 * @author Christian Dühl
 */

public class JarStarter {

    /** Der Dateiname der zu startenden Jar-Datei mit Pfad. */
    private final String jarFilename;

    /** Der ausgeführte Befehl. */
    private String command;

    /** Gibt an, ob der Aufruf erfolgreich war. */
    private boolean success;

    /** Die Parameter, die an die Jar übergeben werden sollen. */
    private List<String> parameters;

    /**
     * Konstruktor.
     *
     * @param jarFilename
     *            Der Dateiname der zu startenden Jar-Datei mit oder ohne Pfad. Falls sie ohne Pfad
     *            übergeben wird, so wird angenommen, dass die Jar im current working directory
     *            liegt.
     * @param parameters
     *            Die Parameter, die dem Aufruf mitgegeben werden.
     */
    public JarStarter(String jarFilename, String ... parameters) {
        if (FileHelper.isBareFilename(jarFilename)) {
            String currentWorkingDirectory = SystemTools.getCurrentWorkingDirectory();
            this.jarFilename = FileHelper.concatPathes(currentWorkingDirectory, jarFilename);
        }
        else {
            this.jarFilename = jarFilename;
        }

        this.parameters = CollectionsHelper.arrayToList(parameters);
    }

    /** Startet die Jar-Datei. */
    public void start() {
        String javaExecutablePath = SystemTools.determineRunningJavaExecutable();
        /*
         * Sowas wie
         *     "C:\\Program Files\\Eclipse Adoptium\\jdk-17.0.1.12-hotspot\\bin\\javaw.exe";
         */
        if (javaExecutablePath.isBlank()) {
            throw new RuntimeException("Die Java-Executable konnte nicht ermittelt werden.");
        }
        command = "\"" + javaExecutablePath + "\"" + " -jar " + jarFilename + createParameterPart();
        CmdStarter starter = new CmdStarter(new PrintErrorHandler());
        String startedParams = starter.runAndForget(command);
        if (startedParams.isBlank()) {
            Text.say("Konnte nicht gestartet werden: " + command);
            success = false;
        }
        else {
            success = true;
        }
    }

    private String createParameterPart() {
        StringBuilder builder = new StringBuilder();

        for (String parameter : parameters) {
            builder.append(" ");
            builder.append(parameter);
        }

        return builder.toString();
    }

    /** Gibt an, ob der Aufruf erfolgreich war. */
    public boolean isSuccess() {
        return success;
    }

    /** Getter für den ausgeführten Befehl. */
    public String getCommand() {
        return command;
    }

}
