package de.duehl.basics.debug;

/*
 * Copyright 2016 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import static org.junit.Assert.*;

import java.util.ArrayList;
import java.util.List;

import org.junit.Test;

import de.duehl.basics.collections.CollectionsHelper;

public class AssureTest {

    @Test (expected = IllegalArgumentException.class)
    public void notNullWithNullObjectAndDescription() {
        String testMe = null;
        Assure.notNull("Der String testMe", testMe);
    }

    @Test
    public void notNullWithNullObjectAndDescriptionCatchException() {
        String testMe = null;
        try {
            Assure.notNull("Der String testMe", testMe);
        }
        catch (Exception exception) {
            String actual = exception.getMessage();
            String expected = "Der String testMe darf nicht null sein!";
            assertEquals(expected, actual);
        }
    }

    @Test
    public void notNullWithNotNullObjectAndDescription() {
        String testMe = "foo";
        Assure.notNull("Der String testMe", testMe);
    }

    @Test (expected = IllegalArgumentException.class)
    public void notNullWithNullObject() {
        String testMe = null;
        Assure.notNull(testMe);
    }

    @Test
    public void notNullWithNotNullObject() {
        String testMe = "foo";
        Assure.notNull(testMe);
    }

    @Test
    public void notNullWithNullObjectCatchException() {
        String testMe = null;
        try {
            Assure.notNull(testMe);
        }
        catch (Exception exception) {
            String actual = exception.getMessage();
            String expected = "Das übergebene Objekt darf nicht null sein!";
            assertEquals(expected, actual);
        }
    }

    @Test (expected = IllegalArgumentException.class)
    public void integerIsInRangeFailsWithDescription() {
        int minute = 62;
        Assure.isInRange("Minute", minute, 0, 59);
    }

    @Test
    public void integerIsInRangeSuccessWithDescription() {
        int minute = 13;
        Assure.isInRange("Minute", minute, 0, 59);
    }

    @Test
    public void integerIsInRangeFailsWithDescriptionCatchException() {
        int minute = 62;
        try {
            Assure.isInRange("Minute", minute, 0, 59);
        }
        catch (Exception exception) {
            String actual = exception.getMessage();
            String expected = "Minute liegt nicht im zulässigen Bereich!\n\t"
                    + "Wert                    : 62\n\t"
                    + "Minimaler Wert inclusive: 0\n\t"
                    + "Maximaler Wert inclusive: 59";
            assertEquals(expected, actual);
        }
    }

    @Test (expected = IllegalArgumentException.class)
    public void integerIsInRangeFails() {
        int minute = 62;
        Assure.isInRange(minute, 0, 59);
    }

    @Test
    public void integerIsInRangeSuccess() {
        int minute = 13;
        Assure.isInRange(minute, 0, 59);
    }

    @Test
    public void integerIsInRangeFailsCatchException() {
        int minute = 62;
        try {
            Assure.isInRange(minute, 0, 59);
        }
        catch (Exception exception) {
            String actual = exception.getMessage();
            String expected = "Der übergebene Wert liegt nicht im zulässigen Bereich!\n\t"
                    + "Wert                    : 62\n\t"
                    + "Minimaler Wert inclusive: 0\n\t"
                    + "Maximaler Wert inclusive: 59";
            assertEquals(expected, actual);
        }
    }

    @Test (expected = IllegalArgumentException.class)
    public void integerIsAtLeastFailsWithDescription() {
        int minute = -1;
        Assure.isAtLeast("Minute", minute, 0);
    }

    @Test
    public void integerIsAtLeastSuccessWithDescription() {
        int minute = 13;
        Assure.isAtLeast("Minute", minute, 0);
    }

    @Test
    public void integerIsAtLeastFailsWithDescriptionCatchException() {
        int minute = -1;
        try {
            Assure.isAtLeast("Minute", minute, 0);
        }
        catch (Exception exception) {
            String actual = exception.getMessage();
            String expected = "Minute ist zu klein!\n\t"
                    + "Wert                    : -1\n\t"
                    + "Minimaler Wert inclusive: 0";
            assertEquals(expected, actual);
        }
    }

    @Test (expected = IllegalArgumentException.class)
    public void integerIsAtLeastFails() {
        int minute = -1;
        Assure.isAtLeast(minute, 0);
    }

    @Test
    public void integerIsAtLeastSuccess() {
        int minute = 13;
        Assure.isAtLeast(minute, 0);
    }

    @Test
    public void integerIsAtLeastFailsCatchException() {
        int minute = -1;
        try {
            Assure.isAtLeast(minute, 0);
        }
        catch (Exception exception) {
            String actual = exception.getMessage();
            String expected = "Der übergebene Wert ist zu klein!\n\t"
                    + "Wert                    : -1\n\t"
                    + "Minimaler Wert inclusive: 0";
            assertEquals(expected, actual);
        }
    }

    @Test (expected = IllegalArgumentException.class)
    public void integerIsAtMostFailsWithDescription() {
        int minute = 62;
        Assure.isAtMost("Minute", minute, 59);
    }

    @Test
    public void integerIsAtMostSuccessWithDescription() {
        int minute = 13;
        Assure.isAtMost("Minute", minute, 59);
    }

    @Test
    public void integerIsAtMostFailsWithDescriptionCatchException() {
        int minute = 62;
        try {
            Assure.isAtMost("Minute", minute, 59);
        }
        catch (Exception exception) {
            String actual = exception.getMessage();
            String expected = "Minute ist zu groß!\n\t"
                    + "Wert                    : 62\n\t"
                    + "Maximaler Wert inclusive: 59";
            assertEquals(expected, actual);
        }
    }

    @Test (expected = IllegalArgumentException.class)
    public void integerIsAtMostFails() {
        int minute = 62;
        Assure.isAtMost(minute, 59);
    }

    @Test
    public void integerIsAtMostSuccess() {
        int minute = 13;
        Assure.isAtMost(minute, 59);
    }

    @Test
    public void integerIsAtMostFailsCatchException() {
        int minute = 62;
        try {
            Assure.isAtMost(minute, 59);
        }
        catch (Exception exception) {
            String actual = exception.getMessage();
            String expected = "Der übergebene Wert ist zu groß!\n\t"
                    + "Wert                    : 62\n\t"
                    + "Maximaler Wert inclusive: 59";
            assertEquals(expected, actual);
        }
    }

    @Test (expected = IllegalArgumentException.class)
    public void longIsInRangeFailsWithDescription() {
        long minute = 62L;
        Assure.isInRange("Minute", minute, 0L, 59L);
    }

    @Test
    public void longIsInRangeSuccessWithDescription() {
        long minute = 13L;
        Assure.isInRange("Minute", minute, 0L, 59L);
    }

    @Test
    public void longIsInRangeFailsWithDescriptionCatchException() {
        long minute = 62L;
        try {
            Assure.isInRange("Minute", minute, 0L, 59L);
        }
        catch (Exception exception) {
            String actual = exception.getMessage();
            String expected = "Minute liegt nicht im Zulässigen Bereich!\n\t"
                    + "Wert                    : 62\n\t"
                    + "Minimaler Wert inclusive: 0\n\t"
                    + "Maximaler Wert inclusive: 59";
            assertEquals(expected, actual);
        }
    }

    @Test (expected = IllegalArgumentException.class)
    public void longIsInRangeFails() {
        long minute = 62L;
        Assure.isInRange(minute, 0L, 59L);
    }

    @Test
    public void longIsInRangeSuccess() {
        long minute = 13L;
        Assure.isInRange(minute, 0L, 59L);
    }

    @Test
    public void longIsInRangeFailsCatchException() {
        long minute = 62L;
        try {
            Assure.isInRange(minute, 0L, 59L);
        }
        catch (Exception exception) {
            String actual = exception.getMessage();
            String expected = "Der übergebene Wert liegt nicht im Zulässigen Bereich!\n\t"
                    + "Wert                    : 62\n\t"
                    + "Minimaler Wert inclusive: 0\n\t"
                    + "Maximaler Wert inclusive: 59";
            assertEquals(expected, actual);
        }
    }

    @Test (expected = IllegalArgumentException.class)
    public void longIsAtLeastFailsWithDescription() {
        long minute = -1L;
        Assure.isAtLeast("Minute", minute, 0L);
    }

    @Test
    public void longIsAtLeastSuccessWithDescription() {
        long minute = 13L;
        Assure.isAtLeast("Minute", minute, 0L);
    }

    @Test
    public void longIsAtLeastFailsWithDescriptionCatchException() {
        long minute = -1L;
        try {
            Assure.isAtLeast("Minute", minute, 0L);
        }
        catch (Exception exception) {
            String actual = exception.getMessage();
            String expected = "Minute ist zu klein!\n\t"
                    + "Wert                    : -1\n\t"
                    + "Minimaler Wert inclusive: 0";
            assertEquals(expected, actual);
        }
    }

    @Test (expected = IllegalArgumentException.class)
    public void longIsAtLeastFails() {
        long minute = -1L;
        Assure.isAtLeast(minute, 0L);
    }

    @Test
    public void longIsAtLeastSuccess() {
        long minute = 13L;
        Assure.isAtLeast(minute, 0L);
    }

    @Test
    public void longIsAtLeastFailsCatchException() {
        long minute = -1L;
        try {
            Assure.isAtLeast(minute, 0L);
        }
        catch (Exception exception) {
            String actual = exception.getMessage();
            String expected = "Der übergebene Wert ist zu klein!\n\t"
                    + "Wert                    : -1\n\t"
                    + "Minimaler Wert inclusive: 0";
            assertEquals(expected, actual);
        }
    }

    @Test (expected = IllegalArgumentException.class)
    public void longIsAtMostFailsWithDescription() {
        long minute = 62L;
        Assure.isAtMost("Minute", minute, 59L);
    }

    @Test
    public void longIsAtMostSuccessWithDescription() {
        long minute = 13L;
        Assure.isAtMost("Minute", minute, 59L);
    }

    @Test
    public void longIsAtMostFailsWithDescriptionCatchException() {
        long minute = 62L;
        try {
            Assure.isAtMost("Minute", minute, 59L);
        }
        catch (Exception exception) {
            String actual = exception.getMessage();
            String expected = "Minute ist zu groß!\n\t"
                    + "Wert                    : 62\n\t"
                    + "Maximaler Wert inclusive: 59";
            assertEquals(expected, actual);
        }
    }

    @Test (expected = IllegalArgumentException.class)
    public void longIsAtMostFails() {
        long minute = 62L;
        Assure.isAtMost(minute, 59L);
    }

    @Test
    public void longIsAtMostSuccess() {
        long minute = 13L;
        Assure.isAtMost(minute, 59L);
    }

    @Test
    public void longIsAtMostFailsCatchException() {
        long minute = 62L;
        try {
            Assure.isAtMost(minute, 59L);
        }
        catch (Exception exception) {
            String actual = exception.getMessage();
            String expected = "Der übergebene Wert ist zu groß!\n\t"
                    + "Wert                    : 62\n\t"
                    + "Maximaler Wert inclusive: 59";
            assertEquals(expected, actual);
        }
    }

    @Test (expected = IllegalArgumentException.class)
    public void floatIsInRangeFailsWithDescription() {
        float probability = 2.5f;
        Assure.isInRange("Wahrscheinlichkeit", probability, 0.0f, 1.0f);
    }

    @Test
    public void floatIsInRangeSuccessWithDescription() {
        float probability = 0.5f;
        Assure.isInRange("Wahrscheinlichkeit", probability, 0.0f, 1.0f);
    }

    @Test
    public void floatIsInRangeFailsWithDescriptionCatchException() {
        float probability = 2.5f;
        try {
            Assure.isInRange("Wahrscheinlichkeit", probability, 0.0f, 1.0f);
        }
        catch (Exception exception) {
            String actual = exception.getMessage();
            String expected = "Wahrscheinlichkeit liegt nicht im Zulässigen Bereich!\n\t"
                    + "Wert                    : 2.5\n\t"
                    + "Minimaler Wert inclusive: 0.0\n\t"
                    + "Maximaler Wert inclusive: 1.0";
            assertEquals(expected, actual);
        }
    }

    @Test (expected = IllegalArgumentException.class)
    public void floatIsInRangeFails() {
        float probability = 2.5f;
        Assure.isInRange(probability, 0.0f, 1.0f);
    }

    @Test
    public void floatIsInRangeSuccess() {
        float probability = 0.5f;
        Assure.isInRange(probability, 0.0f, 1.0f);
    }

    @Test
    public void floatIsInRangeFailsCatchException() {
        float probability = 2.5f;
        try {
            Assure.isInRange(probability, 0.0f, 1.0f);
        }
        catch (Exception exception) {
            String actual = exception.getMessage();
            String expected = "Der übergebene Wert liegt nicht im Zulässigen Bereich!\n\t"
                    + "Wert                    : 2.5\n\t"
                    + "Minimaler Wert inclusive: 0.0\n\t"
                    + "Maximaler Wert inclusive: 1.0";
            assertEquals(expected, actual);
        }
    }

    @Test (expected = IllegalArgumentException.class)
    public void floatIsAtLeastFailsWithDescription() {
        float probability = -1.0f;
        Assure.isAtLeast("Wahrscheinlichkeit", probability, 0.0f);
    }

    @Test
    public void floatIsAtLeastSuccessWithDescription() {
        float probability = 0.5f;
        Assure.isAtLeast("Wahrscheinlichkeit", probability, 0.0f);
    }

    @Test
    public void floatIsAtLeastFailsWithDescriptionCatchException() {
        float probability = -1.0f;
        try {
            Assure.isAtLeast("Wahrscheinlichkeit", probability, 0.0f);
        }
        catch (Exception exception) {
            String actual = exception.getMessage();
            String expected = "Wahrscheinlichkeit ist zu klein!\n\t"
                    + "Wert                    : -1.0\n\t"
                    + "Minimaler Wert inclusive: 0.0";
            assertEquals(expected, actual);
        }
    }

    @Test (expected = IllegalArgumentException.class)
    public void floatIsAtLeastFails() {
        float probability = -1.0f;
        Assure.isAtLeast(probability, 0.0f);
    }

    @Test
    public void floatIsAtLeastSuccess() {
        float probability = 0.5f;
        Assure.isAtLeast(probability, 0.0f);
    }

    @Test
    public void floatIsAtLeastFailsCatchException() {
        float probability = -1.0f;
        try {
            Assure.isAtLeast(probability, 0.0f);
        }
        catch (Exception exception) {
            String actual = exception.getMessage();
            String expected = "Der übergebene Wert ist zu klein!\n\t"
                    + "Wert                    : -1.0\n\t"
                    + "Minimaler Wert inclusive: 0.0";
            assertEquals(expected, actual);
        }
    }

    @Test (expected = IllegalArgumentException.class)
    public void floatIsAtMostFailsWithDescription() {
        float probability = 1.5f;
        Assure.isAtMost("Wahrscheinlichkeit", probability, 1.0f);
    }

    @Test
    public void floatIsAtMostSuccessWithDescription() {
        float probability = 0.5f;
        Assure.isAtMost("Wahrscheinlichkeit", probability, 1.0f);
    }

    @Test
    public void floatIsAtMostFailsWithDescriptionCatchException() {
        float probability = 1.5f;
        try {
            Assure.isAtMost("Wahrscheinlichkeit", probability, 1.0f);
        }
        catch (Exception exception) {
            String actual = exception.getMessage();
            String expected = "Wahrscheinlichkeit ist zu groß!\n\t"
                    + "Wert                    : 1.5\n\t"
                    + "Maximaler Wert inclusive: 1.0";
            assertEquals(expected, actual);
        }
    }

    @Test (expected = IllegalArgumentException.class)
    public void floatIsAtMostFails() {
        float probability = 1.5f;
        Assure.isAtMost(probability, 1.0f);
    }

    @Test
    public void floatIsAtMostSuccess() {
        float probability = 0.5f;
        Assure.isAtMost(probability, 1.0f);
    }

    @Test
    public void floatIsAtMostFailsCatchException() {
        float probability = 1.5f;
        try {
            Assure.isAtMost(probability, 1.0f);
        }
        catch (Exception exception) {
            String actual = exception.getMessage();
            String expected = "Der übergebene Wert ist zu groß!\n\t"
                    + "Wert                    : 1.5\n\t"
                    + "Maximaler Wert inclusive: 1.0";
            assertEquals(expected, actual);
        }
    }

    @Test (expected = IllegalArgumentException.class)
    public void doubleIsInRangeFailsWithDescription() {
        double probability = 2.5d;
        Assure.isInRange("Wahrscheinlichkeit", probability, 0.0d, 1.0d);
    }

    @Test
    public void doubleIsInRangeSuccessWithDescription() {
        double probability = 0.5d;
        Assure.isInRange("Wahrscheinlichkeit", probability, 0.0d, 1.0d);
    }

    @Test
    public void doubleIsInRangeFailsWithDescriptionCatchException() {
        double probability = 2.5d;
        try {
            Assure.isInRange("Wahrscheinlichkeit", probability, 0.0d, 1.0d);
        }
        catch (Exception exception) {
            String actual = exception.getMessage();
            String expected = "Wahrscheinlichkeit liegt nicht im Zulässigen Bereich!\n\t"
                    + "Wert                    : 2.5\n\t"
                    + "Minimaler Wert inclusive: 0.0\n\t"
                    + "Maximaler Wert inclusive: 1.0";
            assertEquals(expected, actual);
        }
    }

    @Test (expected = IllegalArgumentException.class)
    public void doubleIsInRangeFails() {
        double probability = 2.5d;
        Assure.isInRange(probability, 0.0d, 1.0d);
    }

    @Test
    public void doubleIsInRangeSuccess() {
        double probability = 0.5d;
        Assure.isInRange(probability, 0.0d, 1.0d);
    }

    @Test
    public void doubleIsInRangeFailsCatchException() {
        double probability = 2.5d;
        try {
            Assure.isInRange(probability, 0.0d, 1.0d);
        }
        catch (Exception exception) {
            String actual = exception.getMessage();
            String expected = "Der übergebene Wert liegt nicht im Zulässigen Bereich!\n\t"
                    + "Wert                    : 2.5\n\t"
                    + "Minimaler Wert inclusive: 0.0\n\t"
                    + "Maximaler Wert inclusive: 1.0";
            assertEquals(expected, actual);
        }
    }

    @Test (expected = IllegalArgumentException.class)
    public void doubleIsAtLeastFailsWithDescription() {
        double probability = -1.0d;
        Assure.isAtLeast("Wahrscheinlichkeit", probability, 0.0d);
    }

    @Test
    public void doubleIsAtLeastSuccessWithDescription() {
        double probability = 1d;
        Assure.isAtLeast("Wahrscheinlichkeit", probability, 0.0d);
    }

    @Test
    public void doubleIsAtLeastFailsWithDescriptionCatchException() {
        double probability = -1.0d;
        try {
            Assure.isAtLeast("Wahrscheinlichkeit", probability, 0.0d);
        }
        catch (Exception exception) {
            String actual = exception.getMessage();
            String expected = "Wahrscheinlichkeit ist zu klein!\n\t"
                    + "Wert                    : -1.0\n\t"
                    + "Minimaler Wert inclusive: 0.0";
            assertEquals(expected, actual);
        }
    }

    @Test (expected = IllegalArgumentException.class)
    public void doubleIsAtLeastFails() {
        double probability = -1.0d;
        Assure.isAtLeast(probability, 0.0d);
    }

    @Test
    public void doubleIsAtLeastSuccess() {
        double probability = 0.5d;
        Assure.isAtLeast(probability, 0.0d);
    }

    @Test
    public void doubleIsAtLeastFailsCatchException() {
        double probability = -1.0d;
        try {
            Assure.isAtLeast(probability, 0.0d);
        }
        catch (Exception exception) {
            String actual = exception.getMessage();
            String expected = "Der übergebene Wert ist zu klein!\n\t"
                    + "Wert                    : -1.0\n\t"
                    + "Minimaler Wert inclusive: 0.0";
            assertEquals(expected, actual);
        }
    }

    @Test (expected = IllegalArgumentException.class)
    public void doubleIsAtMostFailsWithDescription() {
        double probability = 1.5d;
        Assure.isAtMost("Wahrscheinlichkeit", probability, 1.0d);
    }

    @Test
    public void doubleIsAtMostSuccessWithDescription() {
        double probability = 0.5d;
        Assure.isAtMost("Wahrscheinlichkeit", probability, 1.0d);
    }

    @Test
    public void doubleIsAtMostFailsWithDescriptionCatchException() {
        double probability = 1.5d;
        try {
            Assure.isAtMost("Wahrscheinlichkeit", probability, 1.0d);
        }
        catch (Exception exception) {
            String actual = exception.getMessage();
            String expected = "Wahrscheinlichkeit ist zu groß!\n\t"
                    + "Wert                    : 1.5\n\t"
                    + "Maximaler Wert inclusive: 1.0";
            assertEquals(expected, actual);
        }
    }

    @Test (expected = IllegalArgumentException.class)
    public void doubleIsAtMostFails() {
        double probability = 1.5d;
        Assure.isAtMost(probability, 1.0d);
    }

    @Test
    public void doubleIsAtMostSuccess() {
        double probability = 0.5d;
        Assure.isAtMost(probability, 1.0d);
    }

    @Test
    public void doubleIsAtMostFailsCatchException() {
        double probability = 1.5d;
        try {
            Assure.isAtMost(probability, 1.0d);
        }
        catch (Exception exception) {
            String actual = exception.getMessage();
            String expected = "Der übergebene Wert ist zu groß!\n\t"
                    + "Wert                    : 1.5\n\t"
                    + "Maximaler Wert inclusive: 1.0";
            assertEquals(expected, actual);
        }
    }

    @Test (expected = IllegalArgumentException.class)
    public void integerIsInRangeExclusiveMaximumFailsWithDescription() {
        int minute = 60;
        Assure.isInRangeExclusiveMaximum("Minute", minute, 0, 60);
    }

    @Test
    public void integerIsInRangeExclusiveMaximumSuccessWithDescription() {
        int minute = 13;
        Assure.isInRangeExclusiveMaximum("Minute", minute, 0, 60);
    }

    @Test
    public void integerIsInRangeExclusiveMaximumeFailsWithDescriptionCatchException() {
        int minute = 60;
        try {
            Assure.isInRangeExclusiveMaximum("Minute", minute, 0, 60);
        }
        catch (Exception exception) {
            String actual = exception.getMessage();
            String expected = "Minute liegt nicht im Zulässigen Bereich!\n\t"
                    + "Wert                    : 60\n\t"
                    + "Minimaler Wert inclusive: 0\n\t"
                    + "Maximaler Wert exclusive: 60";
            assertEquals(expected, actual);
        }
    }

    @Test (expected = IllegalArgumentException.class)
    public void integerIsInRangeExclusiveMaximumFails() {
        int minute = 60;
        Assure.isInRangeExclusiveMaximum(minute, 0, 60);
    }

    @Test
    public void integerIsInRangeExclusiveMaximumSuccess() {
        int minute = 13;
        Assure.isInRangeExclusiveMaximum(minute, 0, 60);
    }

    @Test
    public void integerIsInRangeExclusiveMaximumFailsCatchException() {
        int minute = 60;
        try {
            Assure.isInRangeExclusiveMaximum(minute, 0, 60);
        }
        catch (Exception exception) {
            String actual = exception.getMessage();
            String expected = "Der übergebene Wert liegt nicht im Zulässigen Bereich!\n\t"
                    + "Wert                    : 60\n\t"
                    + "Minimaler Wert inclusive: 0\n\t"
                    + "Maximaler Wert exclusive: 60";
            assertEquals(expected, actual);
        }
    }

    @Test (expected = IllegalArgumentException.class)
    public void longIsInRangeExclusiveMaximumFailsWithDescription() {
        long minute = 60L;
        Assure.isInRangeExclusiveMaximum("Minute", minute, 0L, 60L);
    }

    @Test
    public void longIsInRangeExclusiveMaximumSuccessWithDescription() {
        long minute = 13L;
        Assure.isInRangeExclusiveMaximum("Minute", minute, 0L, 60L);
    }

    @Test
    public void longIsInRangeExclusiveMaximumeFailsWithDescriptionCatchException() {
        long minute = 60L;
        try {
            Assure.isInRangeExclusiveMaximum("Minute", minute, 0L, 60L);
        }
        catch (Exception exception) {
            String actual = exception.getMessage();
            String expected = "Minute liegt nicht im Zulässigen Bereich!\n\t"
                    + "Wert                    : 60\n\t"
                    + "Minimaler Wert inclusive: 0\n\t"
                    + "Maximaler Wert exclusive: 60";
            assertEquals(expected, actual);
        }
    }

    @Test (expected = IllegalArgumentException.class)
    public void longIsInRangeExclusiveMaximumFails() {
        long minute = 60L;
        Assure.isInRangeExclusiveMaximum(minute, 0, 60L);
    }

    @Test
    public void longIsInRangeExclusiveMaximumSuccess() {
        long minute = 13L;
        Assure.isInRangeExclusiveMaximum(minute, 0, 60L);
    }

    @Test
    public void longIsInRangeExclusiveMaximumFailsCatchException() {
        long minute = 60L;
        try {
            Assure.isInRangeExclusiveMaximum(minute, 0, 60L);
        }
        catch (Exception exception) {
            String actual = exception.getMessage();
            String expected = "Der übergebene Wert liegt nicht im Zulässigen Bereich!\n\t"
                    + "Wert                    : 60\n\t"
                    + "Minimaler Wert inclusive: 0\n\t"
                    + "Maximaler Wert exclusive: 60";
            assertEquals(expected, actual);
        }
    }

    @Test (expected = IllegalArgumentException.class)
    public void floatIsInRangeExclusiveMaximumFailsWithDescription() {
        float probability = 1.0f;
        Assure.isInRangeExclusiveMaximum("Wahrscheinlichkeit", probability, 0.0f, 1.0f);
    }

    @Test
    public void floatIsInRangeExclusiveMaximumSuccessWithDescription() {
        float probability = 0.5f;
        Assure.isInRangeExclusiveMaximum("Wahrscheinlichkeit", probability, 0.0f, 1.0f);
    }

    @Test
    public void floatIsInRangeExclusiveMaximumeFailsWithDescriptionCatchException() {
        float probability = 1.0f;
        try {
            Assure.isInRangeExclusiveMaximum("Wahrscheinlichkeit", probability, 0.0f, 1.0f);
        }
        catch (Exception exception) {
            String actual = exception.getMessage();
            String expected = "Wahrscheinlichkeit liegt nicht im Zulässigen Bereich!\n\t"
                    + "Wert                    : 1.0\n\t"
                    + "Minimaler Wert inclusive: 0.0\n\t"
                    + "Maximaler Wert exclusive: 1.0";
            assertEquals(expected, actual);
        }
    }

    @Test (expected = IllegalArgumentException.class)
    public void floatIsInRangeExclusiveMaximumFails() {
        float probability = 1.0f;
        Assure.isInRangeExclusiveMaximum(probability, 0.0f, 1.0f);
    }

    @Test
    public void floatIsInRangeExclusiveMaximumSuccess() {
        float probability = 0.5f;
        Assure.isInRangeExclusiveMaximum(probability, 0.0f, 1.0f);
    }

    @Test
    public void floatIsInRangeExclusiveMaximumFailsCatchException() {
        float probability = 1.0f;
        try {
            Assure.isInRangeExclusiveMaximum(probability, 0.0f, 1.0f);
        }
        catch (Exception exception) {
            String actual = exception.getMessage();
            String expected = "Der übergebene Wert liegt nicht im Zulässigen Bereich!\n\t"
                    + "Wert                    : 1.0\n\t"
                    + "Minimaler Wert inclusive: 0.0\n\t"
                    + "Maximaler Wert exclusive: 1.0";
            assertEquals(expected, actual);
        }
    }

    @Test (expected = IllegalArgumentException.class)
    public void doubleIsInRangeExclusiveMaximumFailsWithDescription() {
        double probability = 1.0d;
        Assure.isInRangeExclusiveMaximum("Wahrscheinlichkeit", probability, 0.0d, 1.0d);
    }

    @Test
    public void doubleIsInRangeExclusiveMaximumSuccessWithDescription() {
        double probability = 0.5d;
        Assure.isInRangeExclusiveMaximum("Wahrscheinlichkeit", probability, 0.0d, 1.0d);
    }

    @Test
    public void doubleIsInRangeExclusiveMaximumeFailsWithDescriptionCatchException() {
        double probability = 1.0d;
        try {
            Assure.isInRangeExclusiveMaximum("Wahrscheinlichkeit", probability, 0.0d, 1.0d);
        }
        catch (Exception exception) {
            String actual = exception.getMessage();
            String expected = "Wahrscheinlichkeit liegt nicht im Zulässigen Bereich!\n\t"
                    + "Wert                    : 1.0\n\t"
                    + "Minimaler Wert inclusive: 0.0\n\t"
                    + "Maximaler Wert exclusive: 1.0";
            assertEquals(expected, actual);
        }
    }

    @Test (expected = IllegalArgumentException.class)
    public void doubleIsInRangeExclusiveMaximumFails() {
        double probability = 1.0d;
        Assure.isInRangeExclusiveMaximum(probability, 0.0d, 1.0d);
    }

    @Test
    public void doubleIsInRangeExclusiveMaximumSuccess() {
        double probability = 0.5d;
        Assure.isInRangeExclusiveMaximum(probability, 0.0d, 1.0d);
    }

    @Test
    public void doubleIsInRangeExclusiveMaximumFailsCatchException() {
        double probability = 1.0d;
        try {
            Assure.isInRangeExclusiveMaximum(probability, 0.0d, 1.0d);
        }
        catch (Exception exception) {
            String actual = exception.getMessage();
            String expected = "Der übergebene Wert liegt nicht im Zulässigen Bereich!\n\t"
                    + "Wert                    : 1.0\n\t"
                    + "Minimaler Wert inclusive: 0.0\n\t"
                    + "Maximaler Wert exclusive: 1.0";
            assertEquals(expected, actual);
        }
    }

    @Test (expected = IllegalArgumentException.class)
    public void integerIsAtMostExclusiveExclusiveMaximumFailsWithDescription() {
        int minute = 60;
        Assure.isAtMostExclusive("Minute", minute, 60);
    }

    @Test
    public void integerIsAtMostExclusiveExclusiveMaximumSuccessWithDescription() {
        int minute = 13;
        Assure.isAtMostExclusive("Minute", minute, 60);
    }

    @Test
    public void integerIsAtMostExclusiveExclusiveMaximumeFailsWithDescriptionCatchException() {
        int minute = 60;
        try {
            Assure.isAtMostExclusive("Minute", minute, 60);
        }
        catch (Exception exception) {
            String actual = exception.getMessage();
            String expected = "Minute ist zu groß!\n\t"
                    + "Wert                    : 60\n\t"
                    + "Maximaler Wert exclusive: 60";
            assertEquals(expected, actual);
        }
    }

    @Test (expected = IllegalArgumentException.class)
    public void integerIsAtMostExclusiveExclusiveMaximumFails() {
        int minute = 60;
        Assure.isAtMostExclusive(minute, 60);
    }

    @Test
    public void integerIsAtMostExclusiveExclusiveMaximumSuccess() {
        int minute = 13;
        Assure.isAtMostExclusive(minute, 60);
    }

    @Test
    public void integerIsAtMostExclusiveExclusiveMaximumFailsCatchException() {
        int minute = 60;
        try {
            Assure.isAtMostExclusive(minute, 60);
        }
        catch (Exception exception) {
            String actual = exception.getMessage();
            String expected = "Der übergebene Wert ist zu groß!\n\t"
                    + "Wert                    : 60\n\t"
                    + "Maximaler Wert exclusive: 60";
            assertEquals(expected, actual);
        }
    }

    @Test (expected = IllegalArgumentException.class)
    public void longIsAtMostExclusiveExclusiveMaximumFailsWithDescription() {
        long minute = 60L;
        Assure.isAtMostExclusive("Minute", minute, 60L);
    }

    @Test
    public void longIsAtMostExclusiveExclusiveMaximumSuccessWithDescription() {
        long minute = 13L;
        Assure.isAtMostExclusive("Minute", minute, 60L);
    }

    @Test
    public void longIsAtMostExclusiveExclusiveMaximumeFailsWithDescriptionCatchException() {
        long minute = 60L;
        try {
            Assure.isAtMostExclusive("Minute", minute, 60L);
        }
        catch (Exception exception) {
            String actual = exception.getMessage();
            String expected = "Minute ist zu groß!\n\t"
                    + "Wert                    : 60\n\t"
                    + "Maximaler Wert exclusive: 60";
            assertEquals(expected, actual);
        }
    }

    @Test (expected = IllegalArgumentException.class)
    public void longIsAtMostExclusiveExclusiveMaximumFails() {
        long minute = 60L;
        Assure.isAtMostExclusive(minute, 60L);
    }

    @Test
    public void longIsAtMostExclusiveExclusiveMaximumSuccess() {
        long minute = 13L;
        Assure.isAtMostExclusive(minute, 60L);
    }

    @Test
    public void longIsAtMostExclusiveExclusiveMaximumFailsCatchException() {
        long minute = 60L;
        try {
            Assure.isAtMostExclusive(minute, 60L);
        }
        catch (Exception exception) {
            String actual = exception.getMessage();
            String expected = "Der übergebene Wert ist zu groß!\n\t"
                    + "Wert                    : 60\n\t"
                    + "Maximaler Wert exclusive: 60";
            assertEquals(expected, actual);
        }
    }

    @Test (expected = IllegalArgumentException.class)
    public void floatIsAtMostExclusiveExclusiveMaximumFailsWithDescription() {
        float probability = 1.0f;
        Assure.isAtMostExclusive("Wahrscheinlichkeit", probability, 1.0f);
    }

    @Test
    public void floatIsAtMostExclusiveExclusiveMaximumSuccessWithDescription() {
        float probability = 0.5f;
        Assure.isAtMostExclusive("Wahrscheinlichkeit", probability, 1.0f);
    }

    @Test
    public void floatIsAtMostExclusiveExclusiveMaximumeFailsWithDescriptionCatchException() {
        float probability = 1.0f;
        try {
            Assure.isAtMostExclusive("Wahrscheinlichkeit", probability, 1.0f);
        }
        catch (Exception exception) {
            String actual = exception.getMessage();
            String expected = "Wahrscheinlichkeit ist zu groß!\n\t"
                    + "Wert                    : 1.0\n\t"
                    + "Maximaler Wert exclusive: 1.0";
            assertEquals(expected, actual);
        }
    }

    @Test (expected = IllegalArgumentException.class)
    public void floatIsAtMostExclusiveExclusiveMaximumFails() {
        float probability = 1.0f;
        Assure.isAtMostExclusive(probability, 1.0f);
    }

    @Test
    public void floatIsAtMostExclusiveExclusiveMaximumSuccess() {
        float probability = 0.5f;
        Assure.isAtMostExclusive(probability, 1.0f);
    }

    @Test
    public void floatIsAtMostExclusiveExclusiveMaximumFailsCatchException() {
        float probability = 1.0f;
        try {
            Assure.isAtMostExclusive(probability, 1.0f);
        }
        catch (Exception exception) {
            String actual = exception.getMessage();
            String expected = "Der übergebene Wert ist zu groß!\n\t"
                    + "Wert                    : 1.0\n\t"
                    + "Maximaler Wert exclusive: 1.0";
            assertEquals(expected, actual);
        }
    }

    @Test (expected = IllegalArgumentException.class)
    public void doubleIsAtMostExclusiveExclusiveMaximumFailsWithDescription() {
        double probability = 1.0d;
        Assure.isAtMostExclusive("Wahrscheinlichkeit", probability, 1.0d);
    }

    @Test
    public void doubleIsAtMostExclusiveExclusiveMaximumSuccessWithDescription() {
        double probability = 0.5d;
        Assure.isAtMostExclusive("Wahrscheinlichkeit", probability, 1.0d);
    }

    @Test
    public void doubleIsAtMostExclusiveExclusiveMaximumeFailsWithDescriptionCatchException() {
        double probability = 1.0d;
        try {
            Assure.isAtMostExclusive("Wahrscheinlichkeit", probability, 1.0d);
        }
        catch (Exception exception) {
            String actual = exception.getMessage();
            String expected = "Wahrscheinlichkeit ist zu groß!\n\t"
                    + "Wert                    : 1.0\n\t"
                    + "Maximaler Wert exclusive: 1.0";
            assertEquals(expected, actual);
        }
    }

    @Test (expected = IllegalArgumentException.class)
    public void doubleIsAtMostExclusiveExclusiveMaximumFails() {
        double probability = 1.0d;
        Assure.isAtMostExclusive(probability, 1.0d);
    }

    @Test
    public void doubleIsAtMostExclusiveExclusiveMaximumSuccess() {
        double probability = 0.5d;
        Assure.isAtMostExclusive(probability, 1.0d);
    }

    @Test
    public void doubleIsAtMostExclusiveExclusiveMaximumFailsCatchException() {
        double probability = 1.0d;
        try {
            Assure.isAtMostExclusive(probability, 1.0d);
        }
        catch (Exception exception) {
            String actual = exception.getMessage();
            String expected = "Der übergebene Wert ist zu groß!\n\t"
                    + "Wert                    : 1.0\n\t"
                    + "Maximaler Wert exclusive: 1.0";
            assertEquals(expected, actual);
        }
    }

    @Test (expected = IllegalArgumentException.class)
    public void notEmptyWithNullObjectAndDescription() {
        String value = null;
        Assure.notEmpty("Dieser wichtige Wert", value);
    }

    @Test (expected = IllegalArgumentException.class)
    public void notEmptyWithEmptyStringAndDescription() {
        String value = "";
        Assure.notEmpty("Dieser wichtige Wert", value);
    }

    @Test
    public void notEmptyWithNullObjectAndDescriptionCatchException() {
        String value = null;
        try {
            Assure.notEmpty("Dieser wichtige Wert", value);
        }
        catch (Exception exception) {
            String actual = exception.getMessage();
            String expected = "Dieser wichtige Wert darf nicht null sein!";
            assertEquals(expected, actual);
        }
    }

    @Test
    public void notEmptyWithEmptyStringAndDescriptionCatchException() {
        String value = "";
        try {
            Assure.notEmpty("Dieser wichtige Wert", value);
        }
        catch (Exception exception) {
            String actual = exception.getMessage();
            String expected = "Dieser wichtige Wert darf nicht leer sein!";
            assertEquals(expected, actual);
        }
    }

    @Test (expected = IllegalArgumentException.class)
    public void notEmptyWithNullObject() {
        String value = null;
        Assure.notEmpty(value);
    }

    @Test
    public void notEmptyWithNullObjectCatchException() {
        String value = null;
        try {
            Assure.notEmpty(value);
        }
        catch (Exception exception) {
            String actual = exception.getMessage();
            String expected = "Der übergebene String darf nicht null sein!";
            assertEquals(expected, actual);
        }
    }

    @Test
    public void notEmptyWithEmptyStringCatchException() {
        String value = "";
        try {
            Assure.notEmpty(value);
        }
        catch (Exception exception) {
            String actual = exception.getMessage();
            String expected = "Der übergebene String darf nicht leer sein!";
            assertEquals(expected, actual);
        }
    }

    @Test
    public void notEmptyWithNotWmptyObjectAndDescription() {
        String value = "foo";
        Assure.notEmpty("Dieser wichtige Wert", value);
    }

    @Test
    public void notEmptyWithNotWmptyObject() {
        String value = "foo";
        Assure.notEmpty(value);
    }

    @Test (expected = IllegalArgumentException.class)
    public void notEmptyListWithNullObjectAndDescription() {
        List<String> list = null;
        Assure.notEmpty("Diese wichtige Liste", list);
    }

    @Test
    public void notEmptyListWithNullObjectAndDescriptionCatchException() {
        List<String> list = null;
        try {
            Assure.notEmpty("Diese wichtige Liste", list);
        }
        catch (Exception exception) {
            String actual = exception.getMessage();
            String expected = "Diese wichtige Liste darf nicht null sein!";
            assertEquals(expected, actual);
        }
    }

    @Test
    public void notEmptyListWithEmptyListAndDescriptionCatchException() {
        List<String> list = new ArrayList<>();
        try {
            Assure.notEmpty("Diese wichtige Liste", list);
        }
        catch (Exception exception) {
            String actual = exception.getMessage();
            String expected = "Diese wichtige Liste darf nicht leer sein!";
            assertEquals(expected, actual);
        }
    }

    @Test (expected = IllegalArgumentException.class)
    public void notEmptyListWithNullObject() {
        List<String> list = null;
        Assure.notEmpty(list);
    }

    @Test
    public void notEmptyListWithNullObjectCatchException() {
        List<String> list = null;
        try {
            Assure.notEmpty(list);
        }
        catch (Exception exception) {
            String actual = exception.getMessage();
            String expected = "Die übergebene Liste darf nicht null sein!";
            assertEquals(expected, actual);
        }
    }

    @Test
    public void notEmptyListWithEmptyListCatchException() {
        List<String> list = new ArrayList<>();
        try {
            Assure.notEmpty(list);
        }
        catch (Exception exception) {
            String actual = exception.getMessage();
            String expected = "Die übergebene Liste darf nicht leer sein!";
            assertEquals(expected, actual);
        }
    }

    @Test
    public void notEmptyListWithNotWmptyObjectAndDescription() {
        List<String> list = new ArrayList<>();
        list.add("foo");
        Assure.notEmpty("Dieser wichtige Wert", list);
    }

    @Test
    public void notEmptyListWithNotWmptyObject() {
        List<String> list = new ArrayList<>();
        list.add("foo");
        Assure.notEmpty(list);
    }

    @Test
    public void isPositive() {
        int value = 13;
        Assure.isPositive(value);
    }

    @Test (expected = IllegalArgumentException.class)
    public void isPositiveFailingWithNull() {
        int value = 0;
        Assure.isPositive(value);
    }

    @Test (expected = IllegalArgumentException.class)
    public void isPositiveFailingWithNegativeNumber() {
        int value = -15;
        Assure.isPositive(value);
    }

    @Test
    public void isPositiveWithDescription() {
        int value = 13;
        Assure.isPositive("bla bla", value);
    }

    @Test (expected = IllegalArgumentException.class)
    public void isPositiveWithDescriptionFailingWithNull() {
        int value = 0;
        Assure.isPositive("bla bla", value);
    }

    @Test (expected = IllegalArgumentException.class)
    public void isPositiveWithDescriptionFailingWithNegativeNumber() {
        int value = -15;
        Assure.isPositive("bla bla", value);
    }

    @Test
    public void isNegative() {
        int value = -13;
        Assure.isNegative(value);
    }

    @Test (expected = IllegalArgumentException.class)
    public void isNegativeFailingWithNull() {
        int value = 0;
        Assure.isNegative(value);
    }

    @Test (expected = IllegalArgumentException.class)
    public void isNegativeFailingWithPositiveNumber() {
        int value = 15;
        Assure.isNegative(value);
    }

    @Test
    public void isNegativeWithDescription() {
        int value = -13;
        Assure.isNegative("bla bla", value);
    }

    @Test (expected = IllegalArgumentException.class)
    public void isNegativeWithDescriptionFailingWithNull() {
        int value = 0;
        Assure.isNegative("bla bla", value);
    }

    @Test (expected = IllegalArgumentException.class)
    public void isNegativeWithDescriptionFailingWithPositiveNumber() {
        int value = 15;
        Assure.isNegative("bla bla", value);
    }

    @Test
    public void isPositiveLong() {
        long value = 13;
        Assure.isPositive(value);
    }

    @Test (expected = IllegalArgumentException.class)
    public void isPositiveFailingWithNullLong() {
        long value = 0;
        Assure.isPositive(value);
    }

    @Test (expected = IllegalArgumentException.class)
    public void isPositiveFailingWithNegativeNumberLong() {
        long value = -15;
        Assure.isPositive(value);
    }

    @Test
    public void isPositiveWithDescriptionLong() {
        long value = 13;
        Assure.isPositive("bla bla", value);
    }

    @Test (expected = IllegalArgumentException.class)
    public void isPositiveWithDescriptionFailingWithNullLong() {
        long value = 0;
        Assure.isPositive("bla bla", value);
    }

    @Test (expected = IllegalArgumentException.class)
    public void isPositiveWithDescriptionFailingWithNegativeNumberLong() {
        long value = -15;
        Assure.isPositive("bla bla", value);
    }

    @Test
    public void isNegativeLong() {
        long value = -13;
        Assure.isNegative(value);
    }

    @Test (expected = IllegalArgumentException.class)
    public void isNegativeFailingWithNullLong() {
        long value = 0;
        Assure.isNegative(value);
    }

    @Test (expected = IllegalArgumentException.class)
    public void isNegativeFailingWithPositiveNumberLong() {
        long value = 15;
        Assure.isNegative(value);
    }

    @Test
    public void isNegativeWithDescriptionLong() {
        long value = -13;
        Assure.isNegative("bla bla", value);
    }

    @Test (expected = IllegalArgumentException.class)
    public void isNegativeWithDescriptionFailingWithNullLong() {
        long value = 0;
        Assure.isNegative("bla bla", value);
    }

    @Test (expected = IllegalArgumentException.class)
    public void isNegativeWithDescriptionFailingWithPositiveNumberLong() {
        long value = 15;
        Assure.isNegative("bla bla", value);
    }

    @Test (expected = IllegalArgumentException.class)
    public void hasOneElementFailsOnEmptyList() {
        List<String> list = new ArrayList<>();
        Assure.hasOneElement(list);
    }

    @Test (expected = IllegalArgumentException.class)
    public void hasOneElementWithDescriptionFailsOnEmptyList() {
        List<String> list = new ArrayList<>();
        Assure.hasOneElement("bla bla", list);
    }

    @Test (expected = IllegalArgumentException.class)
    public void hasOneElementFailsOnListWithTwoElements() {
        List<String> list = new ArrayList<>();
        list.add("foo");
        list.add("bar");
        Assure.hasOneElement(list);
    }

    @Test
    public void hasOneElementWithDescriptionSuccessOnListWithOneElement() {
        List<String> list = new ArrayList<>();
        list.add("foo");
        Assure.hasOneElement("bla bla", list);
    }

    @Test
    public void hasOneElementSuccessOnListWithOneElement() {
        List<String> list = new ArrayList<>();
        list.add("foo");
        Assure.hasOneElement(list);
    }

    @Test (expected = IllegalArgumentException.class)
    public void hasOneElementWithDescriptionFailsOnListWithTwoElements() {
        List<String> list = new ArrayList<>();
        list.add("foo");
        list.add("bar");
        Assure.hasOneElement("bla bla", list);
    }

    @Test
    public void isTrue() {
        boolean bool = true;
        Assure.isTrue(bool);
    }

    @Test
    public void isTrueWithDescription() {
        boolean bool = true;
        Assure.isTrue("bla bla", bool);
    }

    @Test (expected = IllegalArgumentException.class)
    public void isTrueFails() {
        boolean bool = false;
        Assure.isTrue(bool);
    }

    @Test (expected = IllegalArgumentException.class)
    public void isTrueWithDescriptionFAild() {
        boolean bool = false;
        Assure.isTrue("bla bla", bool);
    }

    @Test
    public void isFalse() {
        boolean bool = false;
        Assure.isFalse(bool);
    }

    @Test
    public void isFalseWithDescription() {
        boolean bool = false;
        Assure.isFalse("bla bla", bool);
    }

    @Test (expected = IllegalArgumentException.class)
    public void isFalseFails() {
        boolean bool = true;
        Assure.isFalse(bool);
    }

    @Test
    public void isEqualStrings() {
        Assure.isEqual("bla", "bla");
    }

    @Test
    public void isEqualInt() {
        int one = 1;
        Assure.isEqual(1, one);
    }

    @Test (expected = IllegalArgumentException.class)
    public void isEqualIntFails() {
        int one = 1;
        Assure.isEqual(2, one);
    }

    @Test
    public void isEqualDouble() {
        Assure.isEqual(Double.valueOf(15.3), Double.valueOf(15.3));
    }

    @Test (expected = IllegalArgumentException.class)
    public void isEqualDifferentStrings() {
        Assure.isEqual("bla bla", "bla");
    }

    @Test (expected = IllegalArgumentException.class)
    public void isEqualDifferentDoubles() {
        Assure.isEqual(Double.valueOf(15.3), Double.valueOf(15.3000001));
    }

    @Test
    public void isEqualWithDescriptionStrings() {
        Assure.isEqual("Beschreibung", "bla", "bla");
    }

    @Test
    public void isEqualWithDescriptionDouble() {
        Assure.isEqual("Beschreibung", Double.valueOf(15.3), Double.valueOf(15.3));
    }

    @Test (expected = IllegalArgumentException.class)
    public void isEqualWithDescriptionDifferentStrings() {
        Assure.isEqual("Beschreibung", "bla bla", "bla");
    }

    @Test (expected = IllegalArgumentException.class)
    public void isEqualWithDescriptionDifferentDoubles() {
        Assure.isEqual("Beschreibung", Double.valueOf(15.3), Double.valueOf(15.3000001));
    }

    @Test
    public void isNotEqualStrings() {
        Assure.isNotEqual("foo", "bar");
    }

    @Test
    public void isNotEqualInt() {
        int one = 1;
        Assure.isNotEqual(2, one);
    }

    @Test (expected = IllegalArgumentException.class)
    public void isNotEqualIntFails() {
        int one = 1;
        Assure.isNotEqual(1, one);
    }

    @Test
    public void isNotEqualDouble() {
        Assure.isNotEqual(Double.valueOf(15.3), Double.valueOf(15.4));
    }

    @Test (expected = IllegalArgumentException.class)
    public void isNotEqualDifferentStrings() {
        Assure.isNotEqual("bla", "bla");
    }

    @Test
    public void notContains() {
        List<String> list = CollectionsHelper.buildListFrom("foo", "bar", "baz");
        String element = "foobar";
        Assure.notContains(list, element);
    }

    @Test
    public void notContainsEmptyList() {
        List<String> list = new ArrayList<>();
        String element = "foo";
        Assure.notContains(list, element);
    }

    @Test
    public void notContainsEmptyElement() {
        List<String> list = CollectionsHelper.buildListFrom("foo", "bar", "baz");
        String element = "";
        Assure.notContains(list, element);
    }

    @Test (expected = IllegalArgumentException.class)
    public void notContainsFails() {
        List<String> list = CollectionsHelper.buildListFrom("foo", "bar", "baz");
        String element = "foo";
        Assure.notContains(list, element);
    }

    @Test
    public void notContainsWithDescription() {
        List<String> list = CollectionsHelper.buildListFrom("foo", "bar", "baz");
        String element = "foobar";
        Assure.notContains("Beschreibung", list, element);
    }

    @Test
    public void notContainsWithDescriptionEmptyList() {
        List<String> list = new ArrayList<>();
        String element = "foo";
        Assure.notContains("Beschreibung", list, element);
    }

    @Test
    public void notContainsWithDescriptionEmptyElement() {
        List<String> list = CollectionsHelper.buildListFrom("foo", "bar", "baz");
        String element = "";
        Assure.notContains("Beschreibung", list, element);
    }

    @Test (expected = IllegalArgumentException.class)
    public void notContainsWithDescriptionFails() {
        List<String> list = CollectionsHelper.buildListFrom("foo", "bar", "baz");
        String element = "foo";
        Assure.notContains("Beschreibung", list, element);
    }

    @Test
    public void isDigitSequence() {
        String digits = "12345";
        Assure.isDigitSequence(digits);
    }

    @Test (expected = IllegalArgumentException.class)
    public void isDigitSequenceWithEmptyInput() {
        String digits = "";
        Assure.isDigitSequence(digits);
    }

    @Test (expected = IllegalArgumentException.class)
    public void isDigitSequenceWithNegativeInput() {
        String digits = "-15";
        Assure.isDigitSequence(digits);
    }

    @Test (expected = IllegalArgumentException.class)
    public void isDigitSequenceWithDecimalPoint() {
        String digits = "42.1234";
        Assure.isDigitSequence(digits);
    }

    @Test (expected = IllegalArgumentException.class)
    public void isDigitSequenceWithComma() {
        String digits = "5,789";
        Assure.isDigitSequence(digits);
    }

    @Test (expected = IllegalArgumentException.class)
    public void isDigitSequenceWithNoNumberAtAll() {
        String digits = "zweiundvierzigtausendsiebenhundertunddrei";
        Assure.isDigitSequence(digits);
    }

    @Test
    public void isDigitSequenceWithDescription() {
        String digits = "12345";
        Assure.isDigitSequence("Beschreibung", digits);
    }

    @Test (expected = IllegalArgumentException.class)
    public void isDigitSequenceWithDescriptionWithEmptyInput() {
        String digits = "";
        Assure.isDigitSequence("Beschreibung", digits);
    }

    @Test (expected = IllegalArgumentException.class)
    public void isDigitSequenceWithDescriptionWithNegativeInput() {
        String digits = "-15";
        Assure.isDigitSequence("Beschreibung", digits);
    }

    @Test (expected = IllegalArgumentException.class)
    public void isDigitSequenceWithDescriptionWithDecimalPoint() {
        String digits = "42.1234";
        Assure.isDigitSequence("Beschreibung", digits);
    }

    @Test (expected = IllegalArgumentException.class)
    public void isDigitSequenceWithDescriptionWithComma() {
        String digits = "5,789";
        Assure.isDigitSequence("Beschreibung", digits);
    }

    @Test (expected = IllegalArgumentException.class)
    public void isDigitSequenceWithDescriptionWithNoNumberAtAll() {
        String digits = "zweiundvierzigtausendsiebenhundertunddrei";
        Assure.isDigitSequence("Beschreibung", digits);
    }

}
