package de.duehl.basics.io;

/*
 * Copyright 2019 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import static org.junit.Assert.*;

import org.junit.Test;

public class CharsetTest {

    @Test
    public void iso88591() {
        Charset charset = Charset.ISO_8859_1;
        assertEquals("ISO_8859_1", charset.name());
        assertEquals("ISO-8859-1", charset.getCharsetAsString());
        assertEquals("ISO Latin Alphabet No. 1, a.k.a. ISO-LATIN-1", charset.getDescription());
    }

    @Test
    public void detectFromNameString() {
        String name = "ISO_8859_1";
        Charset actual = Charset.detectFromNameString(name);
        assertTrue(Charset.ISO_8859_1 == actual);
    }

    @Test (expected = IllegalArgumentException.class)
    public void detectFromNameStringWithWrongString() {
        String name = "ISO_8859_2";
        Charset.detectFromNameString(name);
    }

    @Test
    public void detectFromCharsetAsString() {
        String characterAsString = "ISO-8859-1";
        Charset actual = Charset.detectFromCharsetAsString(characterAsString);
        assertTrue(Charset.ISO_8859_1 == actual);
    }

    @Test (expected = IllegalArgumentException.class)
    public void detectFromCharsetAsStringWithWrongString() {
        String characterAsString = "ISO_8859_1";
        Charset.detectFromCharsetAsString(characterAsString);
    }

    @Test
    public void createCompleteCharsetDescription() {
        String actual = Charset.createCompleteCharsetDescription();
        String expected = ""
                + "Übersicht über die Charsets:\n"
                + "US_ASCII   - US-ASCII   - Seven-bit ASCII, a.k.a. ISO646-US, a.k.a. the Basic "
                        + "Latin block of the Unicode character set\n"
                + "ISO_8859_1 - ISO-8859-1 - ISO Latin Alphabet No. 1, a.k.a. ISO-LATIN-1\n"
                + "UTF_8      - UTF-8      - Eight-bit UCS Transformation Format\n"
                + "UTF_16BE   - UTF-16BE   - Sixteen-bit UCS Transformation Format, big-endian "
                        + "byte order\n"
                + "UTF_16LE   - UTF-16LE   - Sixteen-bit UCS Transformation Format, little-endian "
                        + "byte order\n"
                + "UTF_16     - UTF-16     - Sixteen-bit UCS Transformation Format, byte order "
                        + "identified by an optional byte-order mark\n";
        assertEquals(expected, actual);
    }

    @Test
    public void createCompleteCharsetDescriptionWithOnlyName() {
        String actual = Charset.createCompleteCharsetDescriptionWithOnlyName();
        String expected = ""
                + "Übersicht über die Charsets:\n"
                + "US_ASCII   - Seven-bit ASCII, a.k.a. ISO646-US, a.k.a. the Basic Latin block "
                        + "of the Unicode character set\n"
                + "ISO_8859_1 - ISO Latin Alphabet No. 1, a.k.a. ISO-LATIN-1\n"
                + "UTF_8      - Eight-bit UCS Transformation Format\n"
                + "UTF_16BE   - Sixteen-bit UCS Transformation Format, big-endian byte order\n"
                + "UTF_16LE   - Sixteen-bit UCS Transformation Format, little-endian byte order\n"
                + "UTF_16     - Sixteen-bit UCS Transformation Format, byte order identified by "
                        + "an optional byte-order mark\n";
        assertEquals(expected, actual);
    }

    @Test
    public void createCompleteCharsetDescriptionWithOnlyCharsetAsString() {
        String actual = Charset.createCompleteCharsetDescriptionWithOnlyCharsetAsString();
        String expected = ""
                + "Übersicht über die Charsets:\n"
                + "US-ASCII   - Seven-bit ASCII, a.k.a. ISO646-US, a.k.a. the Basic Latin block "
                        + "of the Unicode character set\n"
                + "ISO-8859-1 - ISO Latin Alphabet No. 1, a.k.a. ISO-LATIN-1\n"
                + "UTF-8      - Eight-bit UCS Transformation Format\n"
                + "UTF-16BE   - Sixteen-bit UCS Transformation Format, big-endian byte order\n"
                + "UTF-16LE   - Sixteen-bit UCS Transformation Format, little-endian byte order\n"
                + "UTF-16     - Sixteen-bit UCS Transformation Format, byte order identified by "
                        + "an optional byte-order mark\n";

        assertEquals(expected, actual);
    }

}

