package de.duehl.basics.io.walker;

/*
 * Copyright 2016 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import static org.junit.Assert.*;

import java.io.File;
import java.net.URL;

import org.junit.Test;

public class DirWalkerTest {

    private final String path;

    private String path() {
        String className = getClass().getName();
        int dotIndex = className.indexOf(".");
        while (dotIndex > -1) {
            className = className.substring(dotIndex + 1); // endet nicht auf .
            dotIndex = className.indexOf(".");
        }

        URL url = getClass().getResource(className + ".class");
        File f = new File(url.getPath());
        f = new File(f.getParent());
        f = new File(f.getParent());
        f = new File(f.getParent());
        f = new File(f.getParent());
        f = new File(f.getParent());
        f = new File(f.getParent());
        f = new File(f.getParent());
        String p = f.getPath() + File.separator + "data" + File.separator
                + "dirWalkerTest" + File.separator;
        return p;
    }

    public DirWalkerTest() {
        path = path();
    }

    class TestDirWalkerObserver implements DirWalkerObserver {
        private StringBuilder builder = new StringBuilder();

        @Override
        public void visitFile(String fileName) {
            builder.append("file: " + fileName + "\n");
        }

        @Override
        public void visitEmptyDir(String dirName) {
            builder.append("emptyDir: " + dirName + "\n");
        }

        public String getLog() {
            return builder.toString();
        }

        @Override
        public void visitNotEmptyDir(String dirName) {
            builder.append("notEmptyDir: " + dirName + "\n");
        }
    }


    /* -------------------------- TESTS -------------------------- */

    @Test
    public void pathIsDirectory() {
        File file = new File(path);
        assertTrue(file.isDirectory());
    }

    @Test
    public void create() {
        DirWalker walker = new DirWalker(path);
        assertNotNull(walker);
    }

    @Test (expected = DirWalkerException.class)
    public void failOnCreateWithAPathThatIsNoDirectory() {
        DirWalker walker = new DirWalker("foo bar baz gobbel gobbel :::;;;!!!<<>>");
        assertNotNull(walker);
    }

    @Test
    public void walk() {
        DirWalker walker = new DirWalker(path);
        TestDirWalkerObserver observer = new TestDirWalkerObserver();
        walker.addObserver(observer);
        walker.walk();
        String actual = observer.getLog();
        String expected = ""
                + "notEmptyDir: " + path + "\n"
                + "file: " + path + "a.txt\n"
                + "file: " + path + "b.txt\n"
                + "notEmptyDir: " + path + "subFolder1\n"
                + "file: " + path + "subFolder1\\c.txt\n"
                + "notEmptyDir: " + path + "subFolder1\\subsubFolder11\n"
                + "file: " + path + "subFolder1\\subsubFolder11\\d.txt\n"
                + "notEmptyDir: " + path + "subFolder2\n"
                + "file: " + path + "subFolder2\\e.txt\n"
                + "emptyDir: " + path + "subFolder3Empty\n"
                + "notEmptyDir: " + path + "subFolder4EmptyButSubFolders\n"
                + "notEmptyDir: " + path + "subFolder4EmptyButSubFolders\\subsubFolder41\n"
                + "file: " + path + "subFolder4EmptyButSubFolders\\subsubFolder41\\f.txt\n";

        assertEquals(expected, actual);
    }

}
