package de.duehl.swing.ui.windowFocusFrame;

/*
 * Copyright 2016 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.event.WindowEvent;
import java.awt.event.WindowFocusListener;
import java.util.ArrayList;
import java.util.List;

import javax.swing.JFrame;

/**
 * Diese Klasse stellt einen JFrame dar, der Beobachtern mitteilt, wenn das
 * ganze Anwendungsfenster in diesem Frame den Focus verliert oder bekommt.
 *
 * @version 1.03     2013-08-02
 * @author Christian Dühl
 */

public class JWindowFocusFrame extends JFrame implements WindowFocusObservable {

    private static final long serialVersionUID = 2608033940961073037L;

    /** Liste mit den Beobachtern. */
    private List<WindowFocusObserver> observers;

    /** Konstruktor. */
    public JWindowFocusFrame() {
        super();
        observers = new ArrayList<>();

        /* Hier wird auf die Window-Focus-Änderungen reagiert: */
        addWindowFocusListener(new WindowFocusListener() {
            @Override
            public void windowGainedFocus(WindowEvent arg0) {
                notifyObserversFocusGained();
            }
            @Override
            public void windowLostFocus(WindowEvent arg0) {
                notifyObserversFocusLost();
            }
        });

    }

    /**
     * Fügt einen Beobachter hinzu.
     *
     * @param observer
     *            Beobachter
     */
    @Override
    public void addObserver(WindowFocusObserver observer) {
        observers.add(observer);
    }

    /**
     * Entfernt einen Beobachter.
     *
     * @param observer
     *            Beobachter
     */
    @Override
    public void deleteObserver(WindowFocusObserver observer) {
        observers.remove(observer);
    }

    /** Informiert die Beobachter über den Window-Focus-Gewinn. */
    @Override
    public void notifyObserversFocusGained() {
        for (WindowFocusObserver observer : observers) {
            observer.updateFocusGained();
        }
    }

    /** Informiert die Beobachter über den Window-Focus-Verlust. */
    @Override
    public void notifyObserversFocusLost() {
        for (WindowFocusObserver observer : observers) {
            observer.updateFocusLost();
        }
    }

}
