package de.duehl.vocabulary.japanese.common.persistence.data;

import java.util.List;
import java.util.Objects;

import de.duehl.basics.text.Text;

/**
 * Diese Klasse hält die Daten, die eine eigenen Liste beschreiben, nämlich Kategorie,
 * Unterkategorie und den Namen der Liste.
 *
 * @version 1.01     2025-07-03
 * @author Christian Dühl
 */

public class OwnListPersistanceData {

    /** Die Kategorie zu der die eigene Liste gehört. */
    private String category;

    /** Die Unterkategorie zu der die eigene Liste gehört. */
    private String subCategory;

    /** Der Name der eigenen Liste. */
    private String name;

    /** Konstruktor. */
    public OwnListPersistanceData() {
        category = "";
        subCategory = "";
        name = "";
    }

    /**
     * Gibt an,. ob das Objekt mit drei nicht-leeren Werten initialisiert wurde. DAs bedeutet
     * natürlich noch nicht, dass es so eine Liste gibt. Diese könnte sich sogar seit dem letzten
     * Programmstart geändert haben. Das muss man also im Anschluss prüfen, wenn diese Methode wahr
     * zurückgibt.
     */
    public boolean isInitialised() {
        return !category.isBlank() && !subCategory.isBlank() && !name.isBlank();
    }

    /** Getter für die Kategorie zu der die eigene Liste gehört. */
    public String getCategory() {
        return category;
    }

    /** Setter für die Kategorie zu der die eigene Liste gehört. */
    public void setCategory(String category) {
        this.category = category;
    }

    /** Getter für die Unterkategorie zu der die eigene Liste gehört. */
    public String getSubCategory() {
        return subCategory;
    }

    /** Setter für die Unterkategorie zu der die eigene Liste gehört. */
    public void setSubCategory(String subCategory) {
        this.subCategory = subCategory;
    }

    /** Getter für den Namen der eigenen Liste. */
    public String getName() {
        return name;
    }

    /** Setter für den Namen der eigenen Liste. */
    public void setName(String name) {
        this.name = name;
    }

    private static final String PERISTANCE_SEPARATOR = "#;#";

    /** Erzeugt aus den Daten einen String zum persistenten Abspeichern. */
    public String toPersistanceString() {
        return category + PERISTANCE_SEPARATOR + subCategory + PERISTANCE_SEPARATOR + name;
    }

    /** Erzeugt aus einem persistent abgespeicherten String wieder ein Daten-Objekt. */
    public static OwnListPersistanceData createFromPersistanceString(String persistanceSring) {
        List<String> parts = Text.splitBy(persistanceSring, PERISTANCE_SEPARATOR);
        if (parts.size() == 3) {
            int index = 0;
            String category = parts.get(index++);
            String subCategory = parts.get(index++);
            String name = parts.get(index++);
            OwnListPersistanceData data = new OwnListPersistanceData();
            data.setCategory(category);
            data.setSubCategory(subCategory);
            data.setName(name);
            if (data.isInitialised()) {
                return data;
            }
        }

        return new OwnListPersistanceData();
    }

    @Override
    public String toString() {
        return "OwListPersistanceData [category=" + category + ", subCategory=" + subCategory
                + ", name=" + name + "]";
    }

    @Override
    public int hashCode() {
        return Objects.hash(category, name, subCategory);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        OwnListPersistanceData other = (OwnListPersistanceData) obj;
        return Objects.equals(category, other.category) && Objects.equals(name, other.name)
                && Objects.equals(subCategory, other.subCategory);
    }

}
