package de.duehl.vocabulary.japanese.logic.symbol.kana.internal;

import java.util.Map;

import de.duehl.vocabulary.japanese.common.data.InternalAdditionalKanaData;
import de.duehl.vocabulary.japanese.data.symbol.Hiragana;
import de.duehl.vocabulary.japanese.data.symbol.Katakana;

/**
 * Diese Klasse kann zu einem Kana die internen, benutzerabhängigen Daten abrufen.
 *
 * @version 1.01     2025-02-02
 * @author Christian Dühl
 */

public class RealInternalKanaDataRequester implements InternalKanaDataRequester {

    public static final String HIRAGANA_PRE = "HIRAGANA_";
    public static final String KATAKANA_PRE = "KATAKANA_";


    /**
     * Das Verzeichnis der internen Daten zu einem Kana nach dem dem Schlüssel des zugehörigen
     * Kana.
     */
    private final Map<String, InternalAdditionalKanaData> key2InternalKanaDataMap;

    /**
     * Konstruktor.
     *
     * @param key2InternalKanaDataMap
     *            Das Verzeichnis der internen Daten zu einem Kana nach dem Schlüssel des
     *            zugehörigen Kana.
     */
    public RealInternalKanaDataRequester(
            Map<String, InternalAdditionalKanaData> key2InternalKanaDataMap) {
        this.key2InternalKanaDataMap = key2InternalKanaDataMap;
    }

    /**
     * Gibt die internen Daten zum übergebenen Hiragana zurück.
     *
     * Sind zu der Kana keine internen Daten hinterlegt, so wird eine Ausnahme geworfen.
     *
     * @param hiragana
     *            Die Hiragana zu der man die internen Daten ermitteln möchte.
     * @return Die internen Daten zur übergebenen Kana.
     */
    @Override
    public InternalAdditionalKanaData getInternalDataForHiragana(Hiragana hiragana) {
        String key = HIRAGANA_PRE + hiragana.name();
        if (key2InternalKanaDataMap.containsKey(key)) {
            return key2InternalKanaDataMap.get(key);
        }
        else {
            throw new RuntimeException(""
                    + "Das Verzeichnis der internen Daten zu einem Hiragana nach dem dem Schlüssel "
                    + "des zugehörigen Hiragana enthält keinen Eintrag für das übergebene "
                    + "Hiragana.\n"
                    + "\t" + "hiragana: " + hiragana + "\n"
                    + "\t" + "key  : " + key + "\n"
                    );
        }
    }

    /**
     * Gibt die internen Daten zum übergebenen Katakana zurück.
     *
     * Sind zu der Katakana keine internen Daten hinterlegt, so wird eine Ausnahme geworfen.
     *
     * @param hiragana
     *            Die Katakana zu der man die internen Daten ermitteln möchte.
     * @return Die internen Daten zur übergebenen Katakana.
     */
    @Override
    public InternalAdditionalKanaData getInternalDataForKatakana(Katakana katakana) {
        String key = KATAKANA_PRE + katakana.name();
        if (key2InternalKanaDataMap.containsKey(key)) {
            return key2InternalKanaDataMap.get(key);
        }
        else {
            throw new RuntimeException(""
                    + "Das Verzeichnis der internen Daten zu einem Katakana nach dem dem Schlüssel "
                    + "des zugehörigen Katakana enthält keinen Eintrag für das übergebene "
                    + "Katakana.\n"
                    + "\t" + "katakana: " + katakana + "\n"
                    + "\t" + "key  : " + key + "\n"
                    );
        }
    }

}
