package de.duehl.vocabulary.japanese.ui.dialog;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.GridLayout;
import java.awt.Image;
import java.awt.Point;
import java.util.List;

import javax.swing.JButton;
import javax.swing.JPanel;
import javax.swing.JScrollPane;

import de.duehl.swing.ui.GuiTools;
import de.duehl.swing.ui.dialogs.base.NonModalFrameDialogBase;
import de.duehl.vocabulary.japanese.data.Vocable;
import de.duehl.vocabulary.japanese.ui.data.VocablesWithSpecificConstraintFinder;
import de.duehl.vocabulary.japanese.ui.data.VocablesWithSpecificConstraintLister;

/**
 * Diese Klasse stellt eine Reihe von Schaltern dar, um Vokabeln, die einer bestimmten Bedingung
 * (Suchbegriff oder Wortart enthalten) genügen, anzeigen zu können.
 *
 * @version 1.01     2024-10-26
 * @author Christian Dühl
 */

public class AllVocablesWithSpecificConstraintDialog extends NonModalFrameDialogBase {

    private static final Dimension DIALOG_DIMENSION = new Dimension(1500, 1000);


    /** Die Beschreibung der Bedingung, also "Suchbegriff" oder "Wortart". */
    @SuppressWarnings("unused")
    private final String constraintDescription;

    /** Die Liste mit allen möglichen Bedingungen (also alle Suchbegriffe oder alle Wortarten). */
    private final List<String> constraints;

    /**
     * Findet alle Vokabeln, die einer bestimmten Bedingung genügen, also den Suchbegriff enthalten
     * oder die Wortart haben.
     */
    private final VocablesWithSpecificConstraintFinder finder;

    /**
     * Die Beschreibung der Menge von Vokabeln (wie sie der VocabularyListerDialog erwartet, also
     * z.B. "neuste Vokabeln").
     */
    private final String descritpionForLister;

    /**
     * Zeigt die Vokabeln, die einer bestimmten Bedingung genügen, also den Suchbegriff enthalten
     * oder die Wortart haben, als Liste an.
     */
    private final VocablesWithSpecificConstraintLister lister;

    /** Der Panel mit den Buttons. */
    private final JPanel panel;

    /** Die ScrollPane um den Panel mit den Buttons. */
    private final JScrollPane scrollPane;

    /**
     * Konstruktor.
     *
     * @param parentLocation
     *            Position des Rahmens der Oberfläche, vor der dieser Dialog erzeugt wird.
     * @param programImage
     *            Anzuzeigendes ProgrammIcon.
     * @param dialogTitle
     *            Der Titel des Dialogs.
     * @param constraintDescription
     *            Die Beschreibung der Bedingung, also "Suchbegriff" oder "Wortart".
     * @param constraints
     *            Die Liste mit allen möglichen Bedingungen (also alle Suchbegriffe oder alle
     *            Wortarten).
     * @param finder
     *            Findet alle Vokabeln, die einer bestimmten Bedingung genügen, also den
     *            Suchbegriff enthalten oder die Wortart haben.
     * @param descritpionForLister
     *            Die Beschreibung der Menge von Vokabeln (wie sie der VocabularyListerDialog
     *            erwartet, also z.B. "neuste Vokabeln").
     * @param lister
     *            Zeigt die Vokabeln, die einer bestimmten Bedingung genügen, also den Suchbegriff
     *            enthalten oder die Wortart haben, als Liste an.
     */
    public AllVocablesWithSpecificConstraintDialog(Image programImage, Point parentLocation,
            String dialogTitle, String constraintDescription, List<String> constraints,
            VocablesWithSpecificConstraintFinder finder, String descritpionForLister,
            VocablesWithSpecificConstraintLister lister) {
        super(parentLocation, programImage, dialogTitle, DIALOG_DIMENSION);
        addEscapeBehaviour();

        this.constraintDescription = constraintDescription;
        this.constraints = constraints;
        this.finder = finder;
        this.descritpionForLister = descritpionForLister;
        this.lister = lister;

        panel = new JPanel();
        scrollPane = GuiTools.createScrollPane(panel);

        fillDialog();
    }

    /** Baut die Gui auf. */
    @Override
    protected void populateDialog() {
        initElements();

        add(createCenterPart(), BorderLayout.CENTER);
        add(createButtonsPart(),  BorderLayout.SOUTH);

        GuiTools.scrollScrollbarToMinimumLater(scrollPane);
    }

    private void initElements() {
        initPanel();
        fillPanel();
    }

    private void initPanel() {
        panel.setLayout(new GridLayout(0, 3, 5, 5));
    }

    private void fillPanel() {
        for (String constraint : constraints) {
            JButton button = createButton(constraint);
            panel.add(button);
        }
    }

    private JButton createButton(String constraint) {
        //JButton button = new JButton(constraintDescription + ": '" + constraint + "'");
        JButton button = new JButton(constraint);
        GuiTools.biggerFont(button, 5);
        button.addActionListener(e -> listVocablesWithConstraint(constraint));
        return button;
    }

    private void listVocablesWithConstraint(String constraint) {
        List<Vocable> vocables = finder.findVocablesWithSpecificConstraint(constraint);
        lister.listVocablesWithSpecificConstraint(descritpionForLister + " " + constraint,
                vocables);
    }

    private Component createCenterPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());

        panel.add(scrollPane, BorderLayout.CENTER);

        return panel;
    }

    private Component createButtonsPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());

        panel.add(createQuitButton(), BorderLayout.EAST);

        return panel;
    }

    private Component createQuitButton() {
        JButton button = new JButton("Beenden");
        button.addActionListener(e -> quit());
        return button;
    }

    private void quit() {
        closeDialog();
    }

}
