package de.duehl.vocabulary.japanese.ui.dialog.grammar.subdialogs.bar;

import java.awt.BorderLayout;
import java.awt.Component;

import javax.swing.BorderFactory;
import javax.swing.JLabel;
import javax.swing.JPanel;

import de.duehl.swing.ui.GuiTools;
import de.duehl.swing.ui.move.VerticalMoveButtons;
import de.duehl.swing.ui.move.data.MovingGui;
import de.duehl.swing.ui.move.data.VerticalMoveButtonsUser;
import de.duehl.vocabulary.japanese.data.Vocable;

/**
 * Diese Klasse stellt die grafische Oberfläche der Anzeige einer Grundform (Verb oder Adjektiv)
 * zur manuellen Sortierung bei der Erstellung von grammatikalischen Tabellen von Verben bzw.
 * Adjektiven dar.
 *
 * @version 1.01     2025-07-19
 * @author Christian Dühl
 */

public class MovableGrammarBaseFormBar implements VerticalMoveButtonsUser {

    /** Die grafische Oberfläche die Bars bewegt und angeben kann, ob die Bewegung zulässig ist. */
    private final MovingGui<MovableGrammarBaseFormBar> movingGui;

    /** Die Vokabel in Grundform eines Verbs oder Adjektivs. */
    private final Vocable baseFormVocable;

    /** Der Index der Vokabel dieser Bar in der Liste der Vokabel. */
    private final int index;

    /** Der Panel auf dem die Grundform der Vokabel dargestellt wird. */
    private final JPanel panel;

    /** Der Panel mit Buttons zum Verschieben nach oben und unten. */
    private final VerticalMoveButtons moveButtons;

    /**
     * Konstruktor.
     *
     * @param movingGui
     *            Die grafische Oberfläche die Bars bewegt und angeben kann, ob die Bewegung
     *            zulässig ist.
     * @param baseFormVocable
     *            Die Vokabel in Grundform eines Verbs oder Adjektivs.
     * @param index
     *            Der Index der Vokabel dieser Bar in der Liste der Vokabel.
     */
    public MovableGrammarBaseFormBar(MovingGui<MovableGrammarBaseFormBar> movingGui,
            Vocable baseFormVocable, int index) {
        this.movingGui = movingGui;
        this.index = index;
        this.baseFormVocable = baseFormVocable;

        panel = new JPanel();
        moveButtons = new VerticalMoveButtons((VerticalMoveButtonsUser) this);

        fillPanel();
    }

    private void fillPanel() {
        panel.setLayout(new BorderLayout());
        GuiTools.createTitle(panel);

        panel.add(createLabel(), BorderLayout.CENTER);
        panel.add(moveButtons.getPanel(), BorderLayout.EAST);
    }

    private Component createLabel() {
        String text = baseFormVocable.getKanjiKanaRomajiWithJapaneseBraces();
        JLabel label = new JLabel(text);
        GuiTools.biggerFont(label, 10);
        label.setBorder(BorderFactory.createEmptyBorder(0, 15, 0, 15));
        label.setVerticalAlignment(JLabel.CENTER);
        return label;
    }

    /**
     * Gibt an, ob die nutzende Ui das Quell-Verzeichnis mit diesen Buttons nach oben bewegt werden
     * kann.
     */
    @Override
    public boolean canMoveButtonsUserMoveUp() {
        return movingGui.canBarMoveUp(this);
    }

    /**
     * Gibt an, ob die nutzende Ui das Quell-Verzeichnis mit diesen Buttons nach unten bewegt
     * werden kann.
     */
    @Override
    public boolean canMoveButtonsUserMoveDown() {
        return movingGui.canBarMoveDown(this);
    }

    /** Verschiebt das Quell-Verzeichnis mit diesen Buttons an die erste Stelle. */
    @Override
    public void moveMoveButtonsUserToFirst() {
        movingGui.moveBarToFirst(this);
    }

    /** Verschiebt das Quell-Verzeichnis mit diesen Buttons nach oben. */
    @Override
    public void moveMoveButtonsUserUp() {
        movingGui.moveBarUp(this);
    }

    /** Verschiebt das Quell-Verzeichnis mit diesen Buttons nach unten. */
    @Override
    public void moveMoveButtonsUserDown() {
        movingGui.moveBarDown(this);
    }

    /** Verschiebt das Quell-Verzeichnis mit diesen Buttons an die letzte Stelle. */
    @Override
    public void moveMoveButtonsUserToLast() {
        movingGui.moveBarToLast(this);
    }

    /** Getter für die Vokabel in Grundform eines Verbs oder Adjektivs. */
    public Vocable getBaseFormVocable() {
        return baseFormVocable;
    }

    /** Getter für den Panel auf dem die Grundform der Vokabel dargestellt wird. */
    public Component getPanel() {
        return panel;
    }

    /** Getter für den Index der Vokabel dieser Bar in der Liste der Vokabel. */
    public int getIndex() {
        return index;
    }

    /**
     * Setzt die Farben und Darstellung der Buttons abhängig davon, ob sie verschoben werden
     * können.
     */
    public void setUpAndDownButtonColorsAndEnabled() {
        moveButtons.setUpAndDownButtonColorsAndEnabled();
    }

}
